﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/imagebuilder/ImagebuilderRequest.h>
#include <aws/imagebuilder/Imagebuilder_EXPORTS.h>
#include <aws/imagebuilder/model/WorkflowType.h>

#include <utility>

namespace Aws {
namespace imagebuilder {
namespace Model {

/**
 */
class CreateWorkflowRequest : public ImagebuilderRequest {
 public:
  AWS_IMAGEBUILDER_API CreateWorkflowRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateWorkflow"; }

  AWS_IMAGEBUILDER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the workflow to create.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateWorkflowRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The semantic version of this workflow resource. The semantic version syntax
   * adheres to the following rules.</p>  <p>The semantic version has four
   * nodes: &lt;major&gt;.&lt;minor&gt;.&lt;patch&gt;/&lt;build&gt;. You can assign
   * values for the first three, and can filter on all of them.</p> <p>
   * <b>Assignment:</b> For the first three nodes you can assign any positive integer
   * value, including zero, with an upper limit of 2^30-1, or 1073741823 for each
   * node. Image Builder automatically assigns the build number to the fourth
   * node.</p> <p> <b>Patterns:</b> You can use any numeric pattern that adheres to
   * the assignment requirements for the nodes that you can assign. For example, you
   * might choose a software version pattern, such as 1.0.0, or a date, such as
   * 2021.01.01.</p>
   */
  inline const Aws::String& GetSemanticVersion() const { return m_semanticVersion; }
  inline bool SemanticVersionHasBeenSet() const { return m_semanticVersionHasBeenSet; }
  template <typename SemanticVersionT = Aws::String>
  void SetSemanticVersion(SemanticVersionT&& value) {
    m_semanticVersionHasBeenSet = true;
    m_semanticVersion = std::forward<SemanticVersionT>(value);
  }
  template <typename SemanticVersionT = Aws::String>
  CreateWorkflowRequest& WithSemanticVersion(SemanticVersionT&& value) {
    SetSemanticVersion(std::forward<SemanticVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Describes the workflow.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateWorkflowRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Describes what change has been made in this version of the workflow, or what
   * makes this version different from other versions of the workflow.</p>
   */
  inline const Aws::String& GetChangeDescription() const { return m_changeDescription; }
  inline bool ChangeDescriptionHasBeenSet() const { return m_changeDescriptionHasBeenSet; }
  template <typename ChangeDescriptionT = Aws::String>
  void SetChangeDescription(ChangeDescriptionT&& value) {
    m_changeDescriptionHasBeenSet = true;
    m_changeDescription = std::forward<ChangeDescriptionT>(value);
  }
  template <typename ChangeDescriptionT = Aws::String>
  CreateWorkflowRequest& WithChangeDescription(ChangeDescriptionT&& value) {
    SetChangeDescription(std::forward<ChangeDescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Contains the UTF-8 encoded YAML document content for the workflow.
   * Alternatively, you can specify the <code>uri</code> of a YAML document file
   * stored in Amazon S3. However, you cannot specify both properties.</p>
   */
  inline const Aws::String& GetData() const { return m_data; }
  inline bool DataHasBeenSet() const { return m_dataHasBeenSet; }
  template <typename DataT = Aws::String>
  void SetData(DataT&& value) {
    m_dataHasBeenSet = true;
    m_data = std::forward<DataT>(value);
  }
  template <typename DataT = Aws::String>
  CreateWorkflowRequest& WithData(DataT&& value) {
    SetData(std::forward<DataT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>uri</code> of a YAML component document file. This must be an S3
   * URL (<code>s3://bucket/key</code>), and the requester must have permission to
   * access the S3 bucket it points to. If you use Amazon S3, you can specify
   * component content up to your service quota.</p> <p>Alternatively, you can
   * specify the YAML document inline, using the component <code>data</code>
   * property. You cannot specify both properties.</p>
   */
  inline const Aws::String& GetUri() const { return m_uri; }
  inline bool UriHasBeenSet() const { return m_uriHasBeenSet; }
  template <typename UriT = Aws::String>
  void SetUri(UriT&& value) {
    m_uriHasBeenSet = true;
    m_uri = std::forward<UriT>(value);
  }
  template <typename UriT = Aws::String>
  CreateWorkflowRequest& WithUri(UriT&& value) {
    SetUri(std::forward<UriT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) that uniquely identifies the KMS key used to
   * encrypt this workflow resource. This can be either the Key ARN or the Alias ARN.
   * For more information, see <a
   * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#key-id-key-ARN">Key
   * identifiers (KeyId)</a> in the <i>Key Management Service Developer
   * Guide</i>.</p>
   */
  inline const Aws::String& GetKmsKeyId() const { return m_kmsKeyId; }
  inline bool KmsKeyIdHasBeenSet() const { return m_kmsKeyIdHasBeenSet; }
  template <typename KmsKeyIdT = Aws::String>
  void SetKmsKeyId(KmsKeyIdT&& value) {
    m_kmsKeyIdHasBeenSet = true;
    m_kmsKeyId = std::forward<KmsKeyIdT>(value);
  }
  template <typename KmsKeyIdT = Aws::String>
  CreateWorkflowRequest& WithKmsKeyId(KmsKeyIdT&& value) {
    SetKmsKeyId(std::forward<KmsKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags that apply to the workflow resource.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateWorkflowRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateWorkflowRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Unique, case-sensitive identifier you provide to ensure idempotency of the
   * request. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
   * idempotency</a> in the <i>Amazon EC2 API Reference</i>.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateWorkflowRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The phase in the image build process for which the workflow resource is
   * responsible.</p>
   */
  inline WorkflowType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(WorkflowType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline CreateWorkflowRequest& WithType(WorkflowType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Validates the required permissions for the operation and the request
   * parameters, without actually making the request, and provides an error response.
   * Upon a successful request, the error response is
   * <code>DryRunOperationException</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline CreateWorkflowRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::String m_semanticVersion;

  Aws::String m_description;

  Aws::String m_changeDescription;

  Aws::String m_data;

  Aws::String m_uri;

  Aws::String m_kmsKeyId;

  Aws::Map<Aws::String, Aws::String> m_tags;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};

  WorkflowType m_type{WorkflowType::NOT_SET};

  bool m_dryRun{false};
  bool m_nameHasBeenSet = false;
  bool m_semanticVersionHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_changeDescriptionHasBeenSet = false;
  bool m_dataHasBeenSet = false;
  bool m_uriHasBeenSet = false;
  bool m_kmsKeyIdHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
  bool m_typeHasBeenSet = false;
  bool m_dryRunHasBeenSet = false;
};

}  // namespace Model
}  // namespace imagebuilder
}  // namespace Aws
