;;; Tree-il optimizer

;; Copyright (C) 2009, 2010-2015, 2018-2020 Free Software Foundation, Inc.

;;;; This library is free software; you can redistribute it and/or
;;;; modify it under the terms of the GNU Lesser General Public
;;;; License as published by the Free Software Foundation; either
;;;; version 3 of the License, or (at your option) any later version.
;;;; 
;;;; This library is distributed in the hope that it will be useful,
;;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;;; Lesser General Public License for more details.
;;;; 
;;;; You should have received a copy of the GNU Lesser General Public
;;;; License along with this library; if not, write to the Free Software
;;;; Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

;;; Code:

(define-module (language tree-il optimize)
  #:use-module (ice-9 match)
  ;; FIXME: Perhaps allow bootstrap builds to skip fix-letrec, because
  ;; it imports intset, intmap, etc.
  #:use-module (language tree-il fix-letrec)
  #:use-module (system base optimize)
  #:export (optimize
            make-lowerer
            tree-il-optimizations))

(define (make-optimizer opts)
  (define-syntax lookup
    (syntax-rules ()
      ((lookup kw id)
       (lookup kw id id))
      ((lookup kw submodule proc)
       (and (assq-ref opts kw)
            (module-ref (resolve-interface '(language tree-il submodule))
                        'proc)))))
  (let ((verify     (or (lookup #:verify-tree-il? debug verify-tree-il)
                        (lambda (exp) exp)))
        (resolve    (lookup #:resolve-primitives? primitives resolve-primitives))
        (expand     (lookup #:expand-primitives? primitives expand-primitives))
        (letrectify (lookup #:letrectify? letrectify))
        (seal?      (assq-ref opts #:seal-private-bindings?))
        (peval      (lookup #:partial-eval? peval))
        (eta-expand (lookup #:eta-expand? eta-expand)))
    (define-syntax-rule (run-pass! (proc exp arg ...))
      (when proc (set! exp (verify (proc exp arg ...)))))
    (lambda (exp env)
      (verify exp)
      (run-pass! (resolve exp env))
      (run-pass! (expand exp))
      (run-pass! (letrectify exp #:seal-private-bindings? seal?))
      (run-pass! (fix-letrec exp))
      (run-pass! (peval exp env))
      (run-pass! (eta-expand exp))
      exp)))

(define (optimize x env opts)
  ((make-optimizer opts) x env))

(define (tree-il-optimizations)
  (available-optimizations 'tree-il))

(define (make-lowerer optimization-level opts)
  (define (kw-arg-ref args kw default)
    (match (memq kw args)
      ((_ val . _) val)
      (_ default)))
  (define (enabled-for-level? level) (<= level optimization-level))
  (make-optimizer
   (let lp ((all-opts (tree-il-optimizations)))
     (match all-opts
       (() '())
       (((kw level) . all-opts)
        (acons kw (kw-arg-ref opts kw (enabled-for-level? level))
               (lp all-opts)))))))
