## tests/mpria_test.gi -- GAP Input file
MPRIACopyRightString:="\
\n\
MPRIA - GNU Multi-Precision Rational Interval Arithmetic Library\n\
Copyright (C) 2009-2016  Jerome Benoit <jgmbenoit@rezozer.net>\n\
\n\
This file is part of the GNU MPRIA Library.\n\
\n\
The GNU MPRIA Library is free software: you can redistribute it and/or modify it\n\
under the terms of the GNU General Public License as published by\n\
the Free Software Foundation, either version 3 of the License, or\n\
(at your option) any later version.\n\
\n\
The GNU MPRIA Library is distributed in the hope that it will be useful,\n\
but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the\n\
GNU General Public License for more details.\n\
\n\
You should have received a copy of the GNU General Public License\n\
along with the GNU MPRIA Library.  If not, see <http://www.gnu.org/licenses/>.\n\
\n";

if LoadPackage("io") <> true then
	Error("package `IO' not installed. Please install `IO'\n");
fi;

if LoadPackage("float","0.5.14") <> true then
	Error("package `float' not installed. Please install `float'\n");
fi;
SetFloats(MPFR);

Read("mpria_numtheory.gi");

SetAssertionLevel(1);

	MPRIACopyRightCString:=MPRIACopyRightString;
	MPRIACopyRightCString:=ReplacedString(MPRIACopyRightCString,"\n","\n * ");
	MPRIACopyRightCString:=ReplacedString(MPRIACopyRightCString,"* \n","*\n");
	MPRIACopyRightCString[Length(MPRIACopyRightCString)]:='/';
MPRIACopyRightCString:=Concatenation("/*",MPRIACopyRightCString,"\n");

MPRIAMasterFileName:=FilenameFunc(MPRIAMasterFileNameComment);

	CInputFileName:=SplitString(MPRIAMasterFileName,".");
	CInputFileName[Length(CInputFileName)]:="c";
CInputFileName:=JoinStringsWithSeparator(CInputFileName,".");

	CInputGenericName:=SplitString(MPRIAMasterFileName,"/");
	CInputGenericName:=CInputGenericName[Length(CInputGenericName)];
	CInputGenericName:=SplitString(CInputGenericName,".");
CInputGenericName:=CInputGenericName[1];

CInputShortFileName:=Concatenation(CInputGenericName,".c");

TestFunctionTupleTypeName:=ReplacedString(CInputGenericName,"_table","_tuple");

CInputTemporaryFolder:=Directory("/nonexistent");
CInputTemporaryFileName:="";
CInputStream:=OutputTextNone();

MPRIACInputOpen:=function()
	local timeofday,placeofspace,nameofuser,emailofuser;
	timeofday:=[,"/",,"/",," ",,":",,":",," UTC"];
	timeofday{[5,3,1,7,9,11]}:=List(
			DMYhmsSeconds(IO_gettimeofday().tv_sec),
		n->ReplacedString(String(n,2)," ","0"));
	timeofday:=Concatenation(timeofday);
	RemoveFile(CInputFileName);
	if IsBound(GAPInfo.SystemEnvironment.NICKNAME) = true then
		placeofspace:=GAPInfo.SystemEnvironment.NICKNAME;
	else
		placeofspace:=SplitString(IO_gethostname(),".");
		placeofspace:=placeofspace[1];
	fi;
	if IsBound(GAPInfo.SystemEnvironment.GNUSMITHNAME) then
		nameofuser:=GAPInfo.SystemEnvironment.GNUSMITHNAME;
		if IsBound(GAPInfo.SystemEnvironment.GNUSMITHEMAIL) then
			emailofuser:=GAPInfo.SystemEnvironment.GNUSMITHEMAIL;
		fi;
	elif IsBound(GAPInfo.SystemEnvironment.LOGNAME) then
		nameofuser:=GAPInfo.SystemEnvironment.LOGNAME;
		emailofuser:=Concatenation(nameofuser,"@",placeofspace);
	else
		nameofuser:="unknown";
		emailofuser:="unknown@unknown";
	fi;
	CInputTemporaryFolder:=DirectoryTemporary();
	CInputTemporaryFileName:=Filename(CInputTemporaryFolder,CInputShortFileName);
	CInputStream:=OutputTextFile(CInputTemporaryFileName,false);
	SetPrintFormattingStatus(CInputStream,false);
	AppendTo(CInputStream,
		"/* tests/",CInputShortFileName," -- C input file */\n",
		MPRIACopyRightCString,
		"\n");
	AppendTo(CInputStream,
		"/*\n",
		" * generation date: ",timeofday,"\n",
		" * generation place: ",placeofspace,"\n",
		" * generated with: ",MPRIAMasterFileName," (GAP ",GAPInfo.Version,")\n",
		" * generated by: ",nameofuser," <",emailofuser,">","\n",
		" *\n",
		" * comment: ",MPRIAMasterFileNameComment(),"\n",
		" *\n",
		" */\n",
		"\n");
end;

MPRIACInputClose:=function()
	AppendTo(CInputStream,
		"/* End of file tests/",CInputShortFileName," */\n");
	CloseStream(CInputStream);
	CInputStream:=OutputTextNone();
	Exec("mv",CInputTemporaryFileName,CInputFileName); ## dirty trick
	Exec("rmdir",Filename(CInputTemporaryFolder,"")); ## dirty trick
	CInputTemporaryFileName:="";
	CInputTemporaryFolder:=Directory("/nonexistent");
end;

MPRIACInputWriteTestTableLineBEGIN:=function()
	AppendTo(CInputStream,
		TestFunctionTupleTypeName," test_table[]={","\n"
		);
end;

MPRIACInputWriteTestTableLineEND:=function(terminalargument)
	if not(IsString(terminalargument)) then
		Error("terminalargument must be a string");
	fi;
	AppendTo(CInputStream,
		"\t",terminalargument,"}; /* test_table */","\n",
		"\n");
end;

if not(IsBound(NAN)) then
	NAN:=FLOAT.NAN;
fi;

HexStringifyFloat:=function(a)
	local
		precision,mantissa,exponent,
		hexprecision,mprefix,eprefix,
		answer;
	if not(IsFloat(a)) then
		Error("the argument must be a float, got ", a);
	fi;
	precision:=PrecisionFloat(a);
	hexprecision:=QuoInt(precision,4);
	if not(IsZero(RemInt(precision,4))) then
		hexprecision:=hexprecision+1;
	fi;
	answer:=ExtRepOfObj(a);
	mantissa:=answer[1];
	exponent:=answer[2];
	if not(IsZero(mantissa)) then
		exponent:=exponent-1;
		if IsPosInt(mantissa) then
			mprefix:="+";
		else # IsNegInt(mantissa)
			mantissa:=-mantissa;
			mprefix:="-";
		fi;
	else
		mprefix:="+";
	fi;
	mantissa:=String(LowercaseString(HexStringInt(mantissa)),-hexprecision);
	mantissa:=ReplacedString(mantissa," ","0");
	if IsNegInt(exponent) then
		exponent:=-exponent;
		eprefix:="-";
	else
		eprefix:="+";
	fi;
	exponent:=ReplacedString(String(exponent,2)," ","0");
	answer:=Concatenation(
		mprefix,
		"0x",mantissa{[1]},".",mantissa{[2..Length(mantissa)]},
		"p",
		eprefix,exponent);
	return answer;
end;

SeqStringifyListOfNum:=function(candidate)
	local stringify,etadidnac;
	stringify:=function(x)
		if IsRat(x) then
			return Concatenation("\"",String(x),"\"");
		elif IsFloat(x) then
			if IsFinite(x) then
				return Concatenation("\"",String(x),"\"");
#				return Concatenation("\"",HexStringifyFloat(x),"\"");
			elif IsNaN(x) then
				return "\"nan\"";
			else
				return "\"UNPEXPECTED\"";
			fi;
			return Concatenation("UNDEFINED(",String(x),")");
		fi;
		end;
	etadidnac:=Flat(candidate);
	etadidnac:=List(etadidnac,stringify);
	etadidnac:=JoinStringsWithSeparator(etadidnac);
	return etadidnac;
end;

BestConvergentAtLeft:=-1;
BestConvergentAtRight:=+1;
BestConvergentAtAny:=0;
listof_SurroundingMode:=[
	BestConvergentAtAny,
	BestConvergentAtLeft,
	BestConvergentAtRight
	];

stringify_SurroundingMode:=function(surrounding)
	if (surrounding = BestConvergentAtLeft) then
		return "MPRI_SRND_BCAL";
	elif (surrounding = BestConvergentAtRight) then
		return "MPRI_SRND_BCAR";
	elif (surrounding = BestConvergentAtAny) then
		return "MPRI_SRND_BCAA";
	else
		return Concatenation("UNDEFINED(",String(surrounding),")");
	fi;
end;

is_rational_interval:=function(operand)
	local lb_operand,rb_operand;
	lb_operand:=operand[1];
	rb_operand:=operand[2];
	if (IsRat(lb_operand) and IsRat(rb_operand) and not(rb_operand < lb_operand)) then
		return true;
	else
		return false;
	fi;
end;

compute_quadratic_irrational_interval:=function(a,b,c,bits,surrounding)
	local rationalness,nolispe,order,lep,rep,rrep,d,dum;
	if not(IsRat(a)) then
		Error("the first argument must be a rational, got ", a);
	fi;
	if not(IsPosRat(b) or IsZero(b)) then
		Error("the second argument must be a positive rational, got ", b);
	fi;
	if not(IsRat(c) and not(IsZero(c))) then
		Error("the third argument must be a nonzero rational, got ", c);
	fi;
	if not(IsPosInt(bits)) then
		Error("the fourth argument must be a positive integer, got ", bits);
	fi;
	if not(surrounding in listof_SurroundingMode) then
		Error("the third argument must be a SurroundingMode, got ",
			stringify_SurroundingMode(surrounding));
	fi;
	nolispe:=2^(QuoInt(bits+1,2));
	rationalness:=NumeratorRat(b)*DenominatorRat(b);
	rationalness:= RootInt(rationalness)^2 = rationalness ;
	if rationalness then
		a:=(a+Sqrt(b))/c;
		lep:=ContinuedFractionApproximationOfRational(a,1);
		rep:=ContinuedFractionApproximationOfRational(a,2);
	else
		d:=b; a:=a/c; b:=1/c;
		lep:=ContinuedFractionApproximationOfQuadraticIrrational(a,b,d,1);
		rep:=ContinuedFractionApproximationOfQuadraticIrrational(a,b,d,2);
	fi;
	order:=2;
	while (
			not(nolispe < Minimum(DenominatorRat(lep),AbsoluteValue(NumeratorRat(lep)),AbsoluteValue(NumeratorRat(rep)))) and
			(lep <> rep)
			) do
		lep:=rep;
		order:=order+1;
		if rationalness then
			rep:=ContinuedFractionApproximationOfRational(a,order);
		else
			rep:=ContinuedFractionApproximationOfQuadraticIrrational(a,b,d,order);
		fi;
	od;
	if rationalness then
		rrep:=ContinuedFractionApproximationOfRational(a,order+1);
	else
		rrep:=ContinuedFractionApproximationOfQuadraticIrrational(a,b,d,order+1);
	fi;
	if (surrounding = BestConvergentAtRight) then
		if (rep < lep) then
			lep:=rep;
			rep:=rrep;
		fi;
	elif (surrounding = BestConvergentAtLeft) then
		if (lep < rep) then
			lep:=rrep;
		else
				dum:=lep;
			lep:=rep;
			rep:=dum;
		fi;
	else # (surrounding = BestConvergentAtAny)
		if (rep < lep) then
				dum:=lep;
			lep:=rep;
			rep:=dum;
		fi;
	fi;
	Assert(1,(lep <= rep),"corrupted rational interval !\n");
	return [lep,rep];
end;

compute_rational_interval:=function(a,bits,surrounding)
	return compute_quadratic_irrational_interval(a,0,1,bits,surrounding);
end;

compute_float_interval:=function(a,bits,surrounding)
	local answer;
	if not(IsFloat(a)) then
		Error("the first argument must be a float, got ", a);
	fi;
	if not(IsPosInt(bits)) then
		Error("the second argument must be a positive integer, got ", bits);
	fi;
	if not(surrounding in listof_SurroundingMode) then
		Error("the third argument must be a SurroundingMode, got ",
			stringify_SurroundingMode(surrounding));
	fi;
	answer:=ExtRepOfObj(a);
	if not(IsZero(answer[1])) then
		answer:=answer[1]*2^(-1+answer[2]-LogInt(AbsInt(answer[1]),2));
	else
		answer:=0;
	fi;
	return compute_rational_interval(answer,bits,surrounding);
end;

##
## vim:syntax=gap
##
## End of file tests/mpria_test.gi
