unit NewFCtrl;
{ v1.13 with system image list support added for My Inno Setup Extensions }

{ $Id: NewFCtrl.pas,v 1.5 2000/12/16 17:12:58 mlaan Exp $ }

{$R-,I+}

interface

uses
  WinProcs, WinTypes, Messages, SysUtils, Classes, Controls, Graphics, Forms,
  Menus, StdCtrls;

{$IFNDEF VER80}
{$IFNDEF VER90}
{$IFNDEF VER93}
  {$DEFINE Delphi3orHigher}
{$ENDIF}
{$ENDIF}
{$ENDIF}

type
  TNewDirectoryListBox = class;
  TNewDriveComboBox = class;

{ TNewDirectoryListBox }

  TNewDirectoryListBoxBitmaps = (dcClosedBMP, dcOpenedBMP, dcCurrentBMP);

  TNewDirectoryListBox = class(TCustomListBox)
  private
    FDriveCombo: TNewDriveComboBox;
    FDirLabel: TLabel;
    FInSetDir: Boolean;
    FPreserveCase: Boolean;
    FCaseSensitive: Boolean;
    function GetDrive: Char;
    procedure SetDirLabel (Value: TLabel);
    procedure SetDirLabelCaption;
    procedure CMFontChanged (var Message: TMessage); message CM_FONTCHANGED;
    procedure CMColorChanged (var Message: TMessage); message CM_COLORCHANGED;
    procedure CMSysColorChange (var Message: TMessage); message CM_SYSCOLORCHANGE;
    procedure SetDrive (Value: Char);
    procedure DriveChange (NewDrive: Char);
    procedure SetDir(const NewDirectory: string);
    procedure SetDirectory(const NewDirectory: string); virtual;
    procedure ResetItemHeight;
    procedure ReadBitmaps;
    procedure FreeBitmaps;
  protected
    Bitmaps: array[TNewDirectoryListBoxBitmaps, Boolean] of TBitmap;
    FDirectory: string;
    FOnChange: TNotifyEvent;
    procedure Change; virtual;
    procedure DblClick; override;
    procedure CreateWnd; override;
    procedure DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override;
    function  ReadDirectoryNames(const ParentDirectory: string;
      DirectoryList: TStringList): Integer;
    procedure BuildList; virtual;
    procedure KeyPress(var Key: Char); override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function DisplayCase(const S: String): String;
    function FileCompareText(const A, B: String): Integer;
    function GetItemPath(Index: Integer): string;
    procedure OpenCurrent;
    procedure Refresh;
    property Drive: Char read GetDrive write SetDrive;
    property Directory: string read FDirectory write SetDirectory;
    property PreserveCase: Boolean read FPreserveCase;
    property CaseSensitive: Boolean read FCaseSensitive;
  published
    property Align;
    property Color;
    property Columns;
    property Ctl3D;
    property DirLabel: TLabel read FDirLabel write SetDirLabel;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    {$IFDEF Delphi3orHigher}
    property ImeMode;
    property ImeName;
    {$ENDIF}
    property IntegralHeight;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
  end;

{ TNewDriveComboBox }

  TNewTextCase = (tcLowerCase, tcUpperCase);

  TNewDriveComboBoxBitmaps = 0..25;

  TNewDriveComboBox = class(TCustomComboBox)
  private
    FAutoRefresh: Boolean;
    FDirList: TNewDirectoryListBox;
    FDrive: Char;
    FTextCase: TNewTextCase;
    procedure SetDirListBox (Value: TNewDirectoryListBox);
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    procedure CMColorChanged(var Message: TMessage); message CM_COLORCHANGED;
    procedure CMSysColorChange(var Message: TMessage); message CM_SYSCOLORCHANGE;
    procedure SetDrive(NewDrive: Char);
    procedure SetTextCase(NewTextCase: TNewTextCase);
    procedure ReadBitmaps;
    procedure FreeBitmaps;
    procedure ResetItemHeight;
    function MainWindowHook(var Message: TMessage): Boolean;
  protected
    Bitmaps: array[TNewDriveComboBoxBitmaps, Boolean] of TBitmap;
    procedure CreateWnd; override;
    procedure DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override;
    procedure Click; override;
    procedure BuildList; virtual;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Text;
    property Drive: Char read FDrive write SetDrive;
  published
    property AutoRefresh: Boolean read FAutoRefresh write FAutoRefresh default True;
    property Color;
    property Ctl3D;
    property DirList: TNewDirectoryListBox read FDirList write SetDirListBox;
    property DragMode;
    property DragCursor;
    property Enabled;
    property Font;
    {$IFDEF Delphi3orHigher}
    property ImeMode;
    property ImeName;
    {$ENDIF}
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property TextCase: TNewTextCase read FTextCase write SetTextCase default tcLowerCase;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnDropDown;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnStartDrag;
  end;

{ TNewPathLabel }

  TNewPathLabel = class(TCustomLabel)
  protected
    procedure Paint; override;
  public
    constructor Create (AOwner: TComponent); override;
  published
    property Align;
    property Alignment;
    property Caption stored False;
    property Color;
    property DragCursor;
    property DragMode;
    property Enabled;
    property FocusControl;
    property Font;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property Transparent;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
  end;

procedure Register;

implementation

uses
  ShellApi, Commctrl, {$IFNDEF Delphi3orHigher} Ole2, {$ELSE} ActiveX, {$ENDIF}
  Consts, Dialogs;

{$R NEWFCTRL.R32}

procedure Register;
begin
  RegisterComponents ('JR', [TNewDirectoryListBox, TNewDriveComboBox,
    TNewPathLabel]);
end;

type
  TNewDriveType = (dtUnknown, dtNoDrive, dtFloppy, dtFixed, dtNetwork, dtCDROM,
    dtRAM);

const
  DirListMinHeight = 16;
  DriveComboMinHeight = 16;

  MaxResourceNameLength = 32;

  DriveResourceNames: array[TNewDriveType] of PChar =
    ('NFC_HARD', 'NFC_HARD', 'NFC_FLOPPY', 'NFC_HARD', 'NFC_NETWORK', 'NFC_CDROM', 'NFC_RAM');
  DirectoryResourceNames: array[TNewDirectoryListBoxBitmaps] of PChar =
    ('NFC_CLOSEDFOLDER', 'NFC_OPENFOLDER', 'NFC_CURRENTFOLDER');
  NewResourceSuffix = '95';

var
  UseSystemImageList: Boolean;

function SlashSep(const Path, S: String): String;
begin
  if (Path <> '') and
     {$IFDEF Delphi3orHigher}
     (AnsiLastChar(Path)^ <> '\') then
     {$ELSE}
     (Path[Length(Path)] <> '\') then
     {$ENDIF}
    Result := Path + '\' + S
  else
    Result := Path + S;
end;

{ TNewDriveComboBox }

procedure CutFirstDirectory(var S: String);
var
  Root: Boolean;
  P: Integer;
begin
  if S = '\' then
    S := ''
  else
  begin
    if S[1] = '\' then
    begin
      Root := True;
      Delete(S, 1, 1);
    end
    else
      Root := False;
    if S[1] = '.' then
      Delete(S, 1, 4);
    {$IFDEF Delphi3orHigher}
    P := AnsiPos('\', S);
    {$ELSE}
    P := Pos('\', S);
    {$ENDIF}
    if P <> 0 then
    begin
      Delete(S, 1, P);
      S := '...\' + S;
    end
    else
      S := '';
    if Root then
      S := '\' + S;
  end;
end;

function NewMinimizeName(const Filename: String; Canvas: TCanvas;
  MaxLen: Integer): String;
var
  Drive, Dir, Name: String;
begin
  Result := FileName;
  Dir := ExtractFilePath(Result);
  Name := ExtractFileName(Result);

  if (Length(Dir) >= 2) and (Dir[2] = ':') then
  begin
    Drive := Copy(Dir, 1, 2);
    Delete(Dir, 1, 2);
  end
  else
    Drive := '';
  while ((Dir <> '') or (Drive <> '')) and (Canvas.TextWidth(Result) > MaxLen) do
  begin
    if Dir = '\...\' then
    begin
      Drive := '';
      Dir := '...\';
    end
    else if Dir = '' then
      Drive := ''
    else
      CutFirstDirectory(Dir);
    Result := Drive + Dir + Name;
  end;
end;

function VolumeID (DriveChar: Char): string;
var
  OldErrorMode: Integer;
  Buf: array[0..MAX_PATH-1] of Char;
  NotUsed, VolFlags: DWORD;
begin
  OldErrorMode := SetErrorMode(SEM_FAILCRITICALERRORS);
  try
    if GetVolumeInformation(PChar(DriveChar + ':\'), Buf, SizeOf(Buf),
       nil, NotUsed, VolFlags, nil, 0) then
      Result := Buf
    else
      Result := '';
    if DriveChar < 'a' then
      {$IFDEF Delphi3orHigher}
      Result := AnsiUpperCaseFileName(Result)
      {$ELSE}
      Result := AnsiUpperCase(Result)
      {$ENDIF}
    else
      {$IFDEF Delphi3orHigher}
      Result := AnsiLowerCaseFileName(Result);
      {$ELSE}
      Result := AnsiLowerCase(Result);
      {$ENDIF}
  finally
    SetErrorMode(OldErrorMode);
  end;
end;

function NetworkVolume(DriveChar: Char): string;
var
  Buf: array[0..MAX_PATH-1] of Char;
  DriveStr: array[0..2] of Char;
  BufferSize: DWORD;
begin
  BufferSize := SizeOf(Buf);
  DriveStr[0] := UpCase(DriveChar);
  DriveStr[1] := ':';
  DriveStr[2] := #0;
  if WNetGetConnection(DriveStr, Buf, BufferSize) = WN_SUCCESS then begin
    SetString (Result, Buf, BufferSize);
    if DriveChar < 'a' then
      {$IFDEF Delphi3orHigher}
      Result := AnsiUpperCaseFileName(Result)
      {$ELSE}
      Result := AnsiUpperCase(Result)
      {$ENDIF}
    else
      {$IFDEF Delphi3orHigher}
      Result := AnsiLowerCaseFileName(Result);
      {$ELSE}
      Result := AnsiLowerCase(Result);
      {$ENDIF}
  end
  else
    Result := VolumeID(DriveChar);
end;

procedure NewProcessPath (const EditText: string; var Drive: Char;
  var DirPart: string; var FilePart: string);
var
  SaveDir, Root: String;
begin
  SaveDir := GetCurrentDir;
  Drive := SaveDir[1];
  DirPart := EditText;
  if (Length(DirPart) >= 2) and (DirPart[1] = '[') and
     {$IFDEF Delphi3orHigher}
     (AnsiLastChar(DirPart)^ = ']') then
     {$ELSE}
     (DirPart[Length(DirPart)] = ']') then
     {$ENDIF}
    DirPart := Copy(DirPart, 2, Length(DirPart) - 2)
  else
  begin
    Root := ExtractFileDrive(DirPart);
    if Root = '' then
      Root := ExtractFileDrive(SaveDir)
    else
      Delete(DirPart, 1, Length(Root));
    if (Length(Root) >= 2) and (Root[2] = ':') then
      Drive := Root[1]
    else
      Drive := #0;
  end;

  try
    SetCurrentDir (Root);
    FilePart := ExtractFileName (DirPart);
    if Length(DirPart) = (Length(FilePart) + 1) then
      DirPart := '\'
    else if Length(DirPart) > Length(FilePart) then
      SetLength (DirPart, Length(DirPart) - Length(FilePart) - 1)
    else
    begin
      DirPart := GetCurrentDir;
      Delete(DirPart, 1, Length(ExtractFileDrive(DirPart)));
      if DirPart = '' then
        DirPart := '\';
    end;
    if DirPart <> '' then
      ChDir (DirPart);  {first go to our new directory}
    if (FilePart <> '') and not
       (((Pos('*', FilePart) > 0) or (Pos('?', FilePart) > 0)) or
       FileExists(FilePart)) then
    begin
      ChDir(FilePart);
      if Length(DirPart) = 1 then
        DirPart := '\' + FilePart
      else
        DirPart := DirPart + '\' + FilePart;
      FilePart := '';
    end;
    if Drive = #0 then
      DirPart := Root + DirPart;
  finally
    SetCurrentDir (SaveDir);  { restore original directory }
  end;
end;

function GetItemHeight(Font: TFont): Integer;
var
  DC: HDC;
  SaveFont: HFont;
  Metrics: TTextMetric;
begin
  DC := GetDC(0);
  SaveFont := SelectObject(DC, Font.Handle);
  GetTextMetrics(DC, Metrics);
  SelectObject(DC, SaveFont);
  ReleaseDC(0, DC);
  Result := Metrics.tmHeight;
end;

{ TNewDriveComboBox }

constructor TNewDriveComboBox.Create(AOwner: TComponent);
var
  Temp: String;
begin
  inherited Create(AOwner);
  FAutoRefresh := True;  
  Style := csOwnerDrawFixed;
  ReadBitmaps;
  Temp := GetCurrentDir;
  if (Length(Temp) >= 2) and (Temp[2] = ':') then
    FDrive := Temp[1]  { make default drive selected }
  else
    FDrive := #0;
  ResetItemHeight;
  Application.HookMainWindow(MainWindowHook);
end;

destructor TNewDriveComboBox.Destroy;
begin
  Application.UnhookMainWindow(MainWindowHook);
  FreeBitmaps;
  inherited Destroy;
end;

function TNewDriveComboBox.MainWindowHook(var Message: TMessage): Boolean;
const
  DBT_DEVICEARRIVAL = $8000;
  DBT_DEVICEREMOVECOMPLETE = $8004;
  DBT_DEVTYP_VOLUME = $00000002;  // logical volume
type
  PDevBroadcastHdr = ^TDevBroadcastHdr;
  TDevBroadcastHdr = record
    dbch_size: DWORD;
    dbch_devicetype: DWORD;
    dbch_reserved: DWORD;
  end;
begin
  Result := False;
  if FAutoRefresh and (Message.Msg = WM_DEVICECHANGE) and
     ((Message.WParam = DBT_DEVICEARRIVAL) or (Message.WParam = DBT_DEVICEREMOVECOMPLETE)) and
     (PDevBroadcastHdr(Message.LParam).dbch_devicetype = DBT_DEVTYP_VOLUME) then begin
    ReadBitmaps;
    BuildList;
    SetDrive (FDrive);
  end;
end;

procedure TNewDriveComboBox.BuildList;
var
  DriveNum: Integer;
  DriveChar: Char;
  DriveType: TNewDriveType;
  DriveBits: set of 0..25;

  procedure AddDrive(const VolName: string; Obj: TObject);
  begin
    Items.AddObject(Format('%s: %s', [DriveChar, VolName]), Obj);
  end;

begin
  { fill list }
  Clear;
  Longint(DriveBits) := GetLogicalDrives;
  for DriveNum := 0 to 25 do
  begin
    if not (DriveNum in DriveBits) then Continue;
    DriveChar := Char(DriveNum + Ord('a'));

    DriveType := TNewDriveType(GetDriveType(PChar(DriveChar + ':\')));
    if TextCase = tcUpperCase then
      DriveChar := Upcase(DriveChar);

    case DriveType of
      dtFloppy:  Items.AddObject(DriveChar + ':', Pointer(DriveNum));
      dtNetwork: AddDrive(NetworkVolume(DriveChar), Pointer(DriveNum));
    else
      AddDrive(VolumeID(DriveChar), Pointer(DriveNum));
    end;
  end;
end;

procedure TNewDriveComboBox.SetDrive(NewDrive: Char);
var
  SaveDrive: Char;
  Item: Integer;
  drv: string;
begin
  if (ItemIndex < 0) or (UpCase(NewDrive) <> UpCase(FDrive)) then
  begin
    SaveDrive := FDrive;
    try
      if NewDrive = #0 then
      begin
        FDrive := NewDrive;
        ItemIndex := -1;
      end
      else
      begin
        FDrive := UpCase(NewDrive);
        if TextCase = tcLowerCase then
          Inc (FDrive, 32);

        { change selected item }
        for Item := 0 to Items.Count - 1 do
        begin
          drv := Items[Item];
          if (UpCase(drv[1]) = UpCase(FDrive)) and (drv[2] = ':') then
          begin
            ItemIndex := Item;
            break;
          end;
        end;
      end;
      if FDirList <> nil then FDirList.DriveChange(Drive);
      Change;
    except
      FDrive := SaveDrive;
      raise;
    end;
  end;
end;

procedure TNewDriveComboBox.SetTextCase(NewTextCase: TNewTextCase);
var
  OldDrive: Char;
begin
  FTextCase := NewTextCase;
  OldDrive := FDrive;
  BuildList;
  SetDrive (OldDrive);
end;

procedure TNewDriveComboBox.SetDirListBox (Value: TNewDirectoryListBox);
begin
  if FDirList <> nil then FDirList.FDriveCombo := nil;
  FDirList := Value;
  if FDirList <> nil then
  begin
    FDirList.FDriveCombo := Self;
    FDirList.FreeNotification(Self);
  end;
end;

procedure TNewDriveComboBox.CreateWnd;
begin
  inherited CreateWnd;
  BuildList;
  SetDrive (FDrive);
end;

procedure TNewDriveComboBox.DrawItem (Index: Integer; Rect: TRect;
  State: TOwnerDrawState);
var
  Bitmap: TBitmap;
  bmpWidth: Integer;
  R: TRect;
begin
  with Canvas do begin
    FillRect (Rect);

    bmpWidth := 16;
    Bitmap := Bitmaps[TNewDriveComboBoxBitmaps(Items.Objects[Index]),
      odSelected in State];
    if Bitmap <> nil then begin
      bmpWidth := Bitmap.Width;
      Draw (Rect.Left + 4, (Rect.Top + Rect.Bottom - Bitmap.Height) div 2, Bitmap);
    end;

    { uses DrawText instead of TextOut in order to get clipping against
      the combo box button }
    R := Rect;
    Inc (R.Left, bmpWidth + 8);
    DrawText (Canvas.Handle, PChar(Items[Index]), -1, R,
      DT_SINGLELINE or DT_VCENTER or DT_NOPREFIX);
  end;
end;

procedure TNewDriveComboBox.Click;
begin
  inherited Click;
  if ItemIndex >= 0 then begin
    try
      Drive := Items[ItemIndex][1];
    except
      ItemIndex := -1;
      raise;
    end;
  end;
end;

procedure TNewDriveComboBox.CMFontChanged(var Message: TMessage);
begin
  inherited;
  ResetItemHeight;
  RecreateWnd;
end;

procedure TNewDriveComboBox.ResetItemHeight;
var
  H: Integer;
begin
  H := GetItemHeight(Font);
  if H < DriveComboMinHeight then H := DriveComboMinHeight;
  ItemHeight := H;
end;

procedure TNewDriveComboBox.CMColorChanged(var Message: TMessage);
begin
  ReadBitmaps;
  inherited;
end;

procedure TNewDriveComboBox.CMSysColorChange(var Message: TMessage);
begin
  ReadBitmaps;
  inherited;
end;

procedure TNewDriveComboBox.ReadBitmaps;
var
  Temp: TBitmap;
  Buffer: array[0..MaxResourceNameLength-1] of Char;
  I: Integer;
  R: TRect;
  DriveChar: Char;
  DriveBits: set of 0..25;
  SystemImageListHandle: THandle;
  SystemImageListCX, SystemImageListCY: Integer;
  FileInfo: TSHFileInfo;
  BkColor: TColorRef;
begin
  SystemImageListHandle := 0;
  if UseSystemImageList then begin
    SystemImageListHandle := SHGetFileInfo('', 0, FileInfo, SizeOf(FileInfo), SHGFI_SYSICONINDEX or SHGFI_SMALLICON or SHGFI_USEFILEATTRIBUTES);
    if SystemImageListHandle <> 0 then
      ImageList_GetIconSize(SystemImageListHandle, SystemImageListCX, SystemImageListCY);
  end;

  FreeBitmaps;

  { assign bitmap glyphs }
  Temp := TBitmap.Create;

  if UseSystemImageList and (SystemImageListHandle <> 0) then begin
    Temp.Width := SystemImageListCX;
    Temp.Height := SystemImageListCY;
  end;

  try
    Longint(DriveBits) := GetLogicalDrives;
    for I := 0 to 25 do begin
      if not (I in DriveBits) then Continue;
      DriveChar := Char(I + Ord('a'));
      if UseSystemImageList and (SystemImageListHandle <> 0) then begin
        SHGetFileInfo(PChar(DriveChar + ':\'), 0, FileInfo, SizeOf(FileInfo), SHGFI_SYSICONINDEX or SHGFI_SMALLICON);
        BkColor := ImageList_GetBkColor(SystemImageListHandle);
        ImageList_SetBkColor(SystemImageListHandle, ColorToRGB(Self.Color));
        ImageList_Draw(SystemImageListHandle, FileInfo.iIcon, Temp.Canvas.Handle, 0, 0, ILD_NORMAL);
        Bitmaps[I, False] := TBitmap.Create;
        Bitmaps[I, False].Assign(Temp);
        ImageList_SetBkColor(SystemImageListHandle, ColorToRGB(clHighlight));
        ImageList_Draw(SystemImageListHandle, FileInfo.iIcon, Temp.Canvas.Handle, 0, 0, ILD_NORMAL);
        Bitmaps[I, True] := TBitmap.Create;
        Bitmaps[I, True].Assign(Temp);
        ImageList_SetBkColor(SystemImageListHandle, BkColor);
      end else begin
        StrCopy (Buffer, DriveResourceNames[TNewDriveType(GetDriveType(PChar(DriveChar + ':\')))]);
        if NewStyleControls then
          StrCat (Buffer, NewResourceSuffix);
        Temp.Handle := LoadBitmap(HInstance, Buffer);
        with Temp do R := Rect(0, 0, Width, Height);
        Bitmaps[I, False] := TBitmap.Create;
        with Bitmaps[I, False] do begin
          Assign (Temp);
          Canvas.Brush.Color := Self.Color;
          Canvas.BrushCopy (R, Temp, R, Temp.Canvas.Pixels[0, Temp.Height-1]);
        end;
        Bitmaps[I, True] := TBitmap.Create;
        with Bitmaps[I, True] do begin
          Assign (Temp);
          Canvas.Brush.Color := clHighlight;
          Canvas.BrushCopy (R, Temp, R, Temp.Canvas.Pixels[0, Temp.Height-1]);
        end;
      end;
    end;

  finally
    Temp.Free;
  end;
end;

procedure TNewDriveComboBox.FreeBitmaps;
var
  I: TNewDriveComboBoxBitmaps;
  B: Boolean;
begin
  for I := Low(I) to High(I) do
    for B := False to True do
      if Assigned(Bitmaps[I, B]) then begin
        Bitmaps[I, B].Free;
        Bitmaps[I, B] := nil;
      end;
end;

procedure TNewDriveComboBox.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FDirList) then
    FDirList := nil;
end;

{ TNewDirectoryListBox }

function DirLevel(PathName: String): Integer;
var
  P: PChar;
begin
  Result := 0;
  if (PathName <> '') and
     {$IFDEF Delphi3orHigher}
     (AnsiLastChar(PathName)^ <> '\') then
     {$ELSE}
     (PathName[Length(PathName)] <> '\') then
     {$ENDIF}
    Pathname := PathName + '\';
  if (Length(PathName) >= 2) and (PathName[1] = '\') and (PathName[2] = '\') then
    Result := -3;  { handle UNC names appropriately }
  {$IFDEF Delphi3orHigher}
  P := AnsiStrScan(PChar(PathName), '\');
  {$ELSE}
    P := StrScan(PChar(PathName), '\');
  {$ENDIF}
  while P <> nil do
  begin
    Inc(Result);
    Inc(P);
    {$IFDEF Delphi3orHigher}
    P := AnsiStrScan(P, '\');
    {$ELSE}
    P := StrScan(P, '\');
    {$ENDIF}
  end;
end;

constructor TNewDirectoryListBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width := 145;
  Style := lbOwnerDrawFixed;
  Sorted := False;
  ReadBitmaps;
  FDirectory := GetCurrentDir;  { initially use current dir on default drive }
  ResetItemHeight;
end;

destructor TNewDirectoryListBox.Destroy;
begin
  FreeBitmaps;
  inherited Destroy;
end;

procedure TNewDirectoryListBox.DriveChange(NewDrive: Char);
begin
  if UpCase(NewDrive) <> UpCase(Drive) then
  begin
    if NewDrive <> #0 then
    begin
      ChDir(NewDrive + ':');
      FDirectory := GetCurrentDir;  { store correct directory name }
    end;
    if not FInSetDir then
    begin
      BuildList;
      Change;
    end;
  end;
end;

procedure TNewDirectoryListBox.SetDirLabel (Value: TLabel);
begin
  FDirLabel := Value;
  if Value <> nil then Value.FreeNotification(Self);
  SetDirLabelCaption;
end;

procedure TNewDirectoryListBox.SetDir(const NewDirectory: string);
begin
     { go to old directory first, in case of incomplete pathname
       and curdir changed - probably not necessary }
  SetCurrentDir (FDirectory);

  ChDir(NewDirectory);          { exception raised if invalid dir }
  FDirectory := GetCurrentDir;  { store correct directory name }
  BuildList;
  Change;
end;

procedure TNewDirectoryListBox.OpenCurrent;
begin
  Directory := GetItemPath(ItemIndex);
end;

procedure TNewDirectoryListBox.Refresh;
begin
  BuildList;
  Change;
end;

function TNewDirectoryListBox.DisplayCase(const S: String): String;
begin
  if (S = '') or
     (FPreserveCase and ((S[Length(S)] = '\') or (S <> AnsiUpperCase(S)))) or
     FCaseSensitive then
    Result := S
  else
    Result := AnsiLowerCase(S);
end;

function TNewDirectoryListBox.FileCompareText(const A, B: String): Integer;
begin
  if FCaseSensitive then
    Result := AnsiCompareStr(A, B)
  else
    {$IFDEF Delphi3orHigher}
    Result := AnsiCompareFileName(A, B);
    {$ELSE}
    Result := AnsiCompareText(A,B);
    {$ENDIF}
end;

  {
    Reads all directories in ParentDirectory, adds their paths to
    DirectoryList,and returns the number added
  }
function TNewDirectoryListBox.ReadDirectoryNames(const ParentDirectory: string;
  DirectoryList: TStringList): Integer;
var
  SearchRec: TSearchRec;
begin
  Result := 0;
  if FindFirst(SlashSep(ParentDirectory, '*.*'), faDirectory or faSysFile,
     SearchRec) = 0 then
  begin
    try
      repeat
        if (SearchRec.Attr and faDirectory = faDirectory) then
        begin
          if (SearchRec.Name <> '.') and (SearchRec.Name <> '..') then
          begin
            DirectoryList.Add(SearchRec.Name);
            Inc(Result);
          end;
        end;
      until FindNext(SearchRec) <> 0;
    finally
      FindClose(SearchRec);
    end;
  end;
end;

procedure TNewDirectoryListBox.BuildList;
var
  TempPath: String;
  DirName: String;
  IndentLevel, BackSlashPos, i: Integer;
  VolFlags, NotUsed: DWORD;
  Siblings: TStringList;
  NewSelect: Integer;
  Root: String;
begin
  Items.BeginUpdate;
  try
    Items.Clear;
    Root := ExtractFileDrive(Directory) + '\';
    if not GetVolumeInformation(PChar(Root), nil, 0, nil, NotUsed, VolFlags, nil, 0) then
      Exit;
    FPreserveCase := VolFlags and (FS_CASE_IS_PRESERVED or FS_CASE_SENSITIVE) <> 0;
    FCaseSensitive := (VolFlags and FS_CASE_SENSITIVE) <> 0;
    IndentLevel := 0;
    if (Length(Root) >= 2) and (Root[2] = '\') then  { UNC name }
    begin
      TempPath := Copy(Directory, Length(Root)+1, Maxint);
      if TempPath = '' then
        Items.AddObject(Root, Pointer(dcCurrentBMP))
      else
        Items.AddObject(Root, Pointer(dcOpenedBMP));
      Inc(IndentLevel);
    end
    else
      TempPath := Directory;
    if TempPath <> '' then
    begin
      {$IFDEF Delphi3orHigher}
      if AnsiLastChar(TempPath)^ <> '\' then
      {$ELSE}
      if TempPath[Length(TempPath)] <> '\' then
      {$ENDIF}
      begin
        {$IFDEF Delphi3orHigher}
        BackSlashPos := AnsiPos('\', TempPath);
        {$ELSE}
        BackSlashPos := Pos('\', TempPath);
        {$ENDIF}
        while BackSlashPos <> 0 do
        begin
          DirName := Copy(TempPath, 1, BackSlashPos - 1);
          if IndentLevel = 0 then DirName := DirName + '\';
          Delete(TempPath, 1, BackSlashPos);
          Items.AddObject(DirName, Pointer(dcOpenedBMP));

          Inc(IndentLevel);
          {$IFDEF Delphi3orHigher}
          BackSlashPos := AnsiPos('\', TempPath);
          {$ELSE}
          BackSlashPos := Pos('\', TempPath);
          {$ENDIF}
        end;
      end;
      Items.AddObject(TempPath, Pointer(dcCurrentBMP));
    end;
    NewSelect := Items.Count - 1;
    Siblings := TStringList.Create;
    try
      Siblings.Sorted := True;
        { read all the dir names into Siblings }
      ReadDirectoryNames(Directory, Siblings);
      for i := 0 to Siblings.Count - 1 do
        Items.AddObject(Siblings[i], Pointer(dcClosedBMP))
    finally
      Siblings.Free;
    end;
  finally
    Items.EndUpdate;
  end;
  if HandleAllocated then
    ItemIndex := NewSelect;
end;

procedure TNewDirectoryListBox.ReadBitmaps;
var
  Temp: TBitmap;
  I: TNewDirectoryListBoxBitmaps;
  Buffer: array[0..MaxResourceNameLength-1] of Char;
  R: TRect;
  SystemImageListHandle: THandle;
  SystemImageListCX, SystemImageListCY: Integer;
  FileInfo: TSHFileInfo;
  BkColor: TColorRef;
begin
  SystemImageListHandle := 0;
  if UseSystemImageList then begin
    SystemImageListHandle := SHGetFileInfo('', 0, FileInfo, SizeOf(FileInfo), SHGFI_SYSICONINDEX or SHGFI_SMALLICON or SHGFI_USEFILEATTRIBUTES);
    if SystemImageListHandle <> 0 then
      ImageList_GetIconSize(SystemImageListHandle, SystemImageListCX, SystemImageListCY);
  end;

  FreeBitmaps;
  { assign bitmap glyphs }
  Temp := TBitmap.Create;

  if UseSystemImageList and (SystemImageListHandle <> 0) then begin
    Temp.Width := SystemImageListCX;
    Temp.Height := SystemImageListCY;
  end;

  try
    for I := Low(I) to High(I) do begin
      if UseSystemImageList and (SystemImageListHandle <> 0) then begin
        if (I = dcOpenedBMP) or (I = dcCurrentBMP) then
          SHGetFileInfo('c:\directory', FILE_ATTRIBUTE_DIRECTORY, FileInfo, SizeOf(FileInfo), SHGFI_SYSICONINDEX or SHGFI_SMALLICON or SHGFI_USEFILEATTRIBUTES or SHGFI_OPENICON)
        else
          SHGetFileInfo('c:\directory', FILE_ATTRIBUTE_DIRECTORY, FileInfo, SizeOf(FileInfo), SHGFI_SYSICONINDEX or SHGFI_SMALLICON or SHGFI_USEFILEATTRIBUTES);

        BkColor := ImageList_GetBkColor(SystemImageListHandle);
        ImageList_SetBkColor(SystemImageListHandle, ColorToRGB(Self.Color));
        ImageList_Draw(SystemImageListHandle, FileInfo.iIcon, Temp.Canvas.Handle, 0, 0, ILD_NORMAL);
        Bitmaps[I, False] := TBitmap.Create;
        Bitmaps[I, False].Assign(Temp);
        ImageList_SetBkColor(SystemImageListHandle, ColorToRGB(clHighlight));
        ImageList_Draw(SystemImageListHandle, FileInfo.iIcon, Temp.Canvas.Handle, 0, 0, ILD_NORMAL);
        Bitmaps[I, True] := TBitmap.Create;
        Bitmaps[I, True].Assign(Temp);
        ImageList_SetBkColor(SystemImageListHandle, BkColor);
      end else begin
        StrCopy (Buffer, DirectoryResourceNames[I]);
        if NewStyleControls then
          StrCat (Buffer, NewResourceSuffix);
        Temp.Handle := LoadBitmap(HInstance, Buffer);

        with Temp do R := Rect(0, 0, Width, Height);
        Bitmaps[I, False] := TBitmap.Create;
        with Bitmaps[I, False] do begin
          Assign (Temp);
          Canvas.Brush.Color := Self.Color;
          Canvas.BrushCopy (R, Temp, R, Temp.Canvas.Pixels[0, Temp.Height-1]);
        end;
        Bitmaps[I, True] := TBitmap.Create;
        with Bitmaps[I, True] do begin
          Assign (Temp);
          Canvas.Brush.Color := clHighlight;
          Canvas.BrushCopy (R, Temp, R, Temp.Canvas.Pixels[0, Temp.Height-1]);
        end;
      end;
    end;
  finally
    Temp.Free;
  end;
end;

procedure TNewDirectoryListBox.FreeBitmaps;
var
  I: TNewDirectoryListBoxBitmaps;
  B: Boolean;
begin
  for I := Low(I) to High(I) do
    for B := False to True do
      if Assigned(Bitmaps[I, B]) then begin
        Bitmaps[I, B].Free;
        Bitmaps[I, B] := nil;
      end;
end;

procedure TNewDirectoryListBox.DblClick;
begin
  inherited DblClick;
  OpenCurrent;
end;

procedure TNewDirectoryListBox.Change;
begin
  SetDirLabelCaption;
  if Assigned(FOnChange) then FOnChange (Self);
end;

procedure TNewDirectoryListBox.DrawItem(Index: Integer; Rect: TRect;
  State: TOwnerDrawState);
var
  Bitmap: TBitmap;
  bmpWidth: Integer;
  dirOffset: Integer;
  Bmp: TNewDirectoryListBoxBitmaps;
  R: TRect;
begin
  with Canvas do begin
    FillRect (Rect);
    bmpWidth := 16;
    dirOffset := Index * 4 + 4;    {add 4 for spacing}

    Bmp := TNewDirectoryListBoxBitmaps(Items.Objects[Index]);
    Bitmap := TBitmap(Bitmaps[Bmp, odSelected in State]);
    if Bitmap <> nil then begin
      if Bmp = dcClosedBMP then
        dirOffset := DirLevel(Directory) * 4 + 4;

      bmpWidth := Bitmap.Width;
      Draw (Rect.Left + dirOffset, (Rect.Top + Rect.Bottom - Bitmap.Height) div 2,
        Bitmap);
    end;

    R := Classes.Rect(Rect.Left + bmpWidth + dirOffset + 4, Rect.Top, Rect.Right, Rect.Bottom);
    DrawText (Handle, PChar(DisplayCase(Items[Index])), -1,
      R, DT_SINGLELINE or DT_VCENTER or DT_NOPREFIX);
  end;
end;

function TNewDirectoryListBox.GetItemPath (Index: Integer): string;
var
  CurDir: string;
  i, j: Integer;
  Bitmap: TNewDirectoryListBoxBitmaps;
begin
  Result := '';
  if Index < Items.Count then
  begin
    CurDir := Directory;
    Bitmap := TNewDirectoryListBoxBitmaps(Items.Objects[Index]);
    if Index = 0 then
      Result := ExtractFileDrive(CurDir) + '\'
    else
    if Bitmap = dcClosedBMP then
      Result := SlashSep(CurDir, Items[Index])
    else
    if Bitmap = dcCurrentBMP then
      Result := CurDir
    else
    begin
      i := 0;
      j := 0;
      Delete (CurDir, 1, Length(ExtractFileDrive(CurDir)));
      while j <> (Index + 1) do begin
        Inc (i);
        if i > Length(CurDir) then
          break;
        {$IFDEF Delphi3orHigher}
        if CurDir[i] in LeadBytes then
          Inc (i)
        else
        {$ENDIF}
        if CurDir[i] = '\' then
          Inc (j);
      end;
      Result := ExtractFileDrive(Directory) + Copy(CurDir, 1, i - 1);
    end;
  end;
end;

procedure TNewDirectoryListBox.CreateWnd;
begin
  inherited CreateWnd;
  BuildList;
end;

procedure TNewDirectoryListBox.CMFontChanged(var Message: TMessage);
begin
  inherited;
  ResetItemHeight;
end;

procedure TNewDirectoryListBox.ResetItemHeight;
var
  H: Integer;
begin
  H := GetItemHeight(Font);
  if H < DirListMinHeight then H := DirListMinHeight;
  ItemHeight := H;
end;

procedure TNewDirectoryListBox.CMColorChanged(var Message: TMessage);
begin
  ReadBitmaps;
  inherited;
end;

procedure TNewDirectoryListBox.CMSysColorChange(var Message: TMessage);
begin
  ReadBitmaps;
  inherited;
end;

function TNewDirectoryListBox.GetDrive: Char;
begin
  Result := FDirectory[1];
end;

procedure TNewDirectoryListBox.SetDrive(Value: Char);
begin
  if UpCase(Value) <> UpCase(Drive) then
    SetDirectory (Format('%s:', [Value]));
end;

procedure TNewDirectoryListBox.SetDirectory(const NewDirectory: string);
var
  DirPart: string;
  FilePart: string;
  NewDrive: Char;
begin
  if NewDirectory = '' then Exit;
  if (FileCompareText(NewDirectory, Directory) = 0) then Exit;
  NewProcessPath (NewDirectory, NewDrive, DirPart, FilePart);
  try
    if Drive <> NewDrive then
    begin
      FInSetDir := True;
      if (FDriveCombo <> nil) then
        FDriveCombo.Drive := NewDrive
      else
        DriveChange(NewDrive);
    end;
  finally
    FInSetDir := False;
  end;
  SetDir(DirPart);
end;

procedure TNewDirectoryListBox.KeyPress(var Key: Char);
begin
  inherited KeyPress(Key);
  if Ord(Key) = VK_RETURN then
    OpenCurrent;
end;

procedure TNewDirectoryListBox.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) then
  begin
    if (AComponent = FDriveCombo) then FDriveCombo := nil
    else if (AComponent = FDirLabel) then FDirLabel := nil;
  end;
end;

procedure TNewDirectoryListBox.SetDirLabelCaption;
begin
  if FDirLabel <> nil then
    FDirLabel.Caption := Directory;
end;

{ TNewPathLabel }

constructor TNewPathLabel.Create (AOwner: TComponent);
begin
  inherited Create (AOwner);
  AutoSize := False;
  ShowAccelChar := False;
end;

procedure TNewPathLabel.Paint;
const
  Alignments: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);
  WordWraps: array[Boolean] of Word = (0, DT_WORDBREAK);
var
  Rect: TRect;
  Flags: Cardinal;
  S: String;
begin
  S := NewMinimizeName(Caption, Canvas, Width);
  with Canvas do begin
    if not Transparent then begin
      Brush.Color := Self.Color;
      Brush.Style := bsSolid;
      FillRect(ClientRect);
    end;
    Brush.Style := bsClear;
    Rect := ClientRect;
    Flags := DT_EXPANDTABS or DT_NOPREFIX or
      WordWraps[WordWrap] or Alignments[Alignment];
    { Calculate vertical layout }
    Canvas.Font := Self.Font;
    if not Enabled then begin
      OffsetRect(Rect, 1, 1);
      Canvas.Font.Color := clBtnHighlight;
      DrawText(Canvas.Handle, PChar(S), Length(S), Rect, Flags);
      OffsetRect(Rect, -1, -1);
      Canvas.Font.Color := clBtnShadow;
    end;
    DrawText(Canvas.Handle, PChar(S), Length(S), Rect, Flags);
  end;
end;

initialization
  UseSystemImageList := NewStyleControls and SUCCEEDED(CoInitialize(nil));

finalization
  if UseSystemImageList then
    CoUninitialize();

end.


