unit NewGauge;
{ v1.02c }

interface

uses
  WinProcs, WinTypes, SysUtils, Messages, Classes, Graphics, Controls,
  Forms, StdCtrls;

type
  TNewGauge = class(TGraphicControl)
  private
    FMinValue, FMaxValue, FProgress: Longint;
    FShowText: Boolean;
    FForeColor, FBackColor: TColor;
    procedure SetShowText(Value: Boolean);
    procedure SetForeColor(Value: TColor);
    procedure SetBackColor(Value: TColor);
    procedure SetMinValue(Value: Longint);
    procedure SetMaxValue(Value: Longint);
    procedure SetProgress(Value: Longint);
  protected
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property Align;
    property Color;
    property Enabled;
    property ShowText: Boolean read FShowText write SetShowText default True;
    property Font;
    property ForeColor: TColor read FForeColor write SetForeColor default clNavy;
    property BackColor: TColor read FBackColor write SetBackColor default clWhite;
    property MinValue: Longint read FMinValue write SetMinValue;
    property MaxValue: Longint read FMaxValue write SetMaxValue;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property Progress: Longint read FProgress write SetProgress;
    property ShowHint;
    property Visible;
  end;

procedure Register;

implementation

uses
  Consts, NewBevel;

{$IFNDEF VER80}
  {$IFNDEF VER90}
    {$IFNDEF VER93}
      {$DEFINE Delphi3orHigher}
    {$ENDIF}
  {$ENDIF}
{$ENDIF}

procedure Register;
begin
  RegisterComponents('JR', [TNewGauge]);
end;

constructor TNewGauge.Create(AOwner: TComponent);
begin
  inherited Create (AOwner);
  ControlStyle := ControlStyle + [csOpaque];
  { default values }
  FMaxValue := 100;
  FShowText := True;
  FForeColor := clNavy;
  FBackColor := clWhite;
  Width := 100;
  Height := 100;
end;

procedure TNewGauge.Paint;
var
  R, R2: TRect;
  Bmp: TBitmap;
  PercentText: array[0..5] of Char;
  X: Integer;
  procedure DrawPercentText (const DC: HDC);
  var
    SaveIndex: Integer;
  begin
    SaveIndex := SaveDC(DC);
    IntersectClipRect (DC, R2.Left, R2.Top, R2.Right, R2.Bottom);
    DrawText (DC, PercentText, -1, R, DT_CENTER or DT_VCENTER or
      DT_NOCLIP or DT_NOPREFIX or DT_SINGLELINE);
    RestoreDC (DC, SaveIndex);
  end;
begin
  { Border }
  R := ClientRect;
  NewDrawEdge (Canvas, R, deSunken);
  Dec (R.Right, 4);
  Dec (R.Bottom, 4);
  if (R.Right < 1) or (R.Bottom < 1) then Exit;

  Bmp := TBitmap.Create;
  try
    Bmp.Height := R.Bottom;
    Bmp.Width := R.Right;

    try
      if FMaxValue <> FMinValue then
        {$IFNDEF WIN32}
        StrLFmt (PercentText, SizeOf(PercentText)-1, '%.0f %%',
          [((FProgress-FMinValue)/(FMaxValue-FMinValue)) * 100])
        {$ELSE}
        StrLFmt (PercentText, SizeOf(PercentText)-1, '%d %%',
          [MulDiv(100, FProgress-FMinValue, FMaxValue-FMinValue)])
        {$ENDIF}
      else
        StrCopy (PercentText, '0 %');
    except
      StrCopy (PercentText, '?');
    end;

    try
      if FMaxValue <> FMinValue then
        {$IFNDEF WIN32}
        X := Trunc(((FProgress-FMinValue)/(FMaxValue-FMinValue)) * R.Right)
        {$ELSE}
        X := MulDiv(R.Right, FProgress-FMinValue, FMaxValue-FMinValue)
        {$ENDIF}
      else
        X := 0;
    except
      X := 0;
    end;

    with Bmp.Canvas do begin
      R2 := Rect(0, 0, X, R.Bottom);
      Brush.Color := FForeColor;
      FillRect (R2);
      if FShowText then begin
        Font.Assign (Self.Font);
        Font.Color := FBackColor;
        DrawPercentText (Handle);
      end;
      R2 := Rect(X, 0, R.Right, R.Bottom);
      Brush.Color := FBackColor;
      FillRect (R2);
      if FShowText then begin
        Font.Color := FForeColor;
        DrawPercentText (Handle);
      end;
    end;

    Canvas.Draw (2, 2, Bmp);
  finally
    Bmp.Free;
  end;
end;

procedure TNewGauge.SetShowText(Value: Boolean);
begin
  if FShowText <> Value then
  begin
    FShowText := Value;
    Invalidate;
  end;
end;

procedure TNewGauge.SetForeColor(Value: TColor);
begin
  if FForeColor <> Value then begin
    FForeColor := Value;
    Invalidate;
  end;
end;

procedure TNewGauge.SetBackColor(Value: TColor);
begin
  if Value <> FBackColor then
  begin
    FBackColor := Value;
    Invalidate;
  end;
end;

procedure TNewGauge.SetMinValue(Value: Longint);
begin
  if Value <> FMinValue then
  begin
    if Value > FMaxValue then
      raise {$IFNDEF Delphi3orHigher}EInvalidOperation.CreateResFmt{$ELSE}EInvalidOperation.CreateFmt{$ENDIF}
        (SOutOfRange, [-MaxInt, FMaxValue - 1]);
    FMinValue := Value;
    if FProgress < Value then FProgress := Value;
    Invalidate;
  end;
end;

procedure TNewGauge.SetMaxValue(Value: Longint);
begin
  if Value <> FMaxValue then begin
    if Value < FMinValue then
      raise {$IFNDEF Delphi3orHigher}EInvalidOperation.CreateResFmt{$ELSE}EInvalidOperation.CreateFmt{$ENDIF}
        (SOutOfRange, [FMinValue + 1, MaxInt]);
    FMaxValue := Value;
    if FProgress > Value then FProgress := Value;
    Invalidate;
  end;
end;

procedure TNewGauge.SetProgress(Value: Longint);
begin
  if Value < FMinValue then
    Value := FMinValue
  else if Value > FMaxValue then
    Value := FMaxValue;
  if FProgress <> Value then begin
    FProgress := Value;
    Invalidate;
  end;
end;

end.
