unit InstFunc;

{
  Inno Setup
  Copyright (C) 1998-99 Jordan Russell
  For conditions of distribution and use, see LICENSE.TXT.

  Misc. installation functions
}

interface

uses
  WinProcs, WinTypes, SysUtils;

{$I VERSION.INC}

type
  PSimpleStringListArray = ^TSimpleStringListArray;
  TSimpleStringListArray = array[0..$1FFFFFFE] of String;
  TSimpleStringList = class
  private
    FList: PSimpleStringListArray;
    FCount, FCapacity: Integer;
    function Get (Index: Integer): String;
    procedure SetCapacity (NewCapacity: Integer);
  public
    destructor Destroy; override;
    procedure Add (const S: String);
    procedure AddIfDoesntExist (const S: String);
    procedure Clear;
    function IndexOf (const S: String): Integer;

    property Count: Integer read FCount;
    property Items[Index: Integer]: String read Get; default;
  end;

  TDeleteDirProc = function(const DirName: String; Param: Pointer): Boolean;

  TFileVersionNumbers = record
    MS, LS: DWORD;
  end;

const
  RegRootKeyNames: array[HKEY_CLASSES_ROOT..HKEY_DYN_DATA] of PChar = (
    'HKEY_CLASSES_ROOT', 'HKEY_CURRENT_USER', 'HKEY_LOCAL_MACHINE',
    'HKEY_USERS', 'HKEY_PERFORMANCE_DATA', 'HKEY_CURRENT_CONFIG',
    'HKEY_DYN_DATA');

function CheckForMutexes (Mutexes: String): Boolean;
procedure DelayDeleteFile (const Filename: String; const Tries: Integer);
function DelTree (const Path: String; const IsDir, DeleteFiles, DeleteSubdirsAlso: Boolean;
  const DeleteDirProc: TDeleteDirProc; const Param: Pointer): Boolean;
function GenerateUniqueName (Path: String; const Extension: String): String;
function GetComputerNameString: String;
function GetFileDateTime (const Filename: string; var DateTime: TFileTime): Boolean;
function GetFileLocalDateTime (const Filename: string; var DateTime: TFileTime): Boolean;
function GetUserNameString: String;
function GetVersionInfo (const Filename: String; var VersionInfo: TVSFixedFileInfo): Boolean;
function GetVersionNumbers (const Filename: String; var VersionNumbers: TFileVersionNumbers): Boolean;
procedure IncrementSharedCount (const Filename: String;
  const AlreadyExisted: Boolean);
function InstExec (const Filename, Params: String; WorkingDir: String;
  const WaitUntilTerminated, WaitUntilIdle: Boolean; const ShowCmd: Integer;
  const ProcessMessagesProc: TProcedure; var ErrorCode: Integer): Boolean;
function InstShellExec (const Filename, Params: String; WorkingDir: String;
  const ShowCmd: Integer; var ErrorCode: Integer): Boolean;
function ModifyPifFile (const Filename: String; const CloseOnExit: Boolean): Boolean;
procedure RestartComputer;
procedure RestartReplace (const TempFile, DestFile: String);

implementation

uses
  ShellApi,
  CmnFunc2,
  Msgs, MsgIDs;

function GenerateUniqueName (Path: String; const Extension: String): String;
  function IntToBase32 (Number: Longint): String;
  const
    Table: array[0..31] of Char = '0123456789ABCDEFGHIJKLMNOPQRSTUV';
  var
    I: Integer;
  begin
    Result := '';
    for I := 0 to 4 do begin
      Insert (Table[Number and 31], Result, 1);
      Number := Number shr 5;
    end;
  end;
var
  Rand, RandOrig: Longint;
begin
  Path := AddBackslash(Path);
  RandOrig := Random($2000000);
  Rand := RandOrig;
  repeat
    Inc (Rand);
    if Rand > $1FFFFFF then Rand := 0;
    if Rand = RandOrig then
      { practically impossible to go through 33 million possibilities,
        but check "just in case"... }
      raise Exception.Create(FmtSetupMessage1(msgErrorTooManyFilesInDir,
        RemoveBackslashUnlessRoot(Path)));
    { Generate a random name }
    Result := Path + 'is-' + IntToBase32(Rand) + Extension;
  until not FileOrDirExists(Result);
end;

procedure RestartReplace (const TempFile, DestFile: String);
{ Renames TempFile to DestFile the next time Windows is started. If DestFile
  already existed, it will be overwritten. If DestFile is '' then TempFile
  will be deleted, however this is only supported by 95/98 and NT, not
  Windows 3.1x. }
var
  WinDir, WinInitFile, TempWinInitFile: String;
  OldF, NewF: TextFile;
  OldFOpened, NewFOpened: Boolean;
  L, L2: String;
  RenameSectionFound, WriteLastLine: Boolean;
  NewDestFile: PChar;
begin
  if not UsingWinNT then begin
    { Because WININIT.INI allows multiple entries with the same name,
      it must manually parse the file instead of using
      WritePrivateProfileString }
    WinDir := GetWinDir;
    WinInitFile := AddBackslash(WinDir) + 'WININIT.INI';
    OldFOpened := False;
    NewFOpened := False;
    try
      try
        if FileExists(WinInitFile) then begin
          TempWinInitFile := GenerateUniqueName(WinDir, '.tmp');
          { Flush Windows' cache for the file first }
          WritePrivateProfileString (nil, nil, nil, StringAsPChar(WinInitFile));
          AssignFile (OldF, WinInitFile);
          FileMode := fmOpenRead or fmShareDenyWrite;  Reset (OldF);
          OldFOpened := True;
        end
        else
          TempWinInitFile := WinInitFile;
        AssignFile (NewF, TempWinInitFile);
        FileMode := fmOpenWrite or fmShareExclusive;  Rewrite (NewF);
        NewFOpened := True;
        RenameSectionFound := False;
        WriteLastLine := False;
        if OldFOpened then
          while not Eof(OldF) do begin
            Readln (OldF, L);
            WriteLastLine := True;
            L2 := Trim(L);
            if (L2 <> '') and (L2[1] = '[') then begin
              if CompareText(L, '[rename]') = 0 then
                RenameSectionFound := True
              else
              if RenameSectionFound then
                Break;
            end;
            Writeln (NewF, L);
            WriteLastLine := False;
          end;
        if not RenameSectionFound then
          Writeln (NewF, '[rename]');
        if DestFile <> '' then
          L2 := GetShortName(DestFile)
        else
          L2 := 'NUL';
        Writeln (NewF, L2 + '=' + GetShortName(TempFile));
        if OldFOpened then begin
          if WriteLastLine then
            Writeln (NewF, L);
          while not Eof(OldF) do begin
            Readln (OldF, L);
            Writeln (NewF, L);
          end;
        end;
      finally
        if NewFOpened then CloseFile (NewF);
        if OldFOpened then CloseFile (OldF);
      end;
    except
      if OldFOpened then DeleteFile (TempWinInitFile);
      raise;
    end;
    if OldFOpened then begin
      AssignFile (OldF, WinInitFile);
      Erase (OldF);
      AssignFile (OldF, TempWinInitFile);
      Rename (OldF, WinInitFile);
    end;
  end
  else begin
    if DestFile <> '' then
      NewDestFile := StringAsPChar(DestFile)
    else
      NewDestFile := nil;
    if not MoveFileEx(StringAsPChar(TempFile), NewDestFile,
       MOVEFILE_DELAY_UNTIL_REBOOT or MOVEFILE_REPLACE_EXISTING) then
      raise Exception.Create(FmtSetupMessage(msgErrorFunctionFailed,
        ['MoveFileEx', IntToStr(GetLastError)]));
  end;
end;

function DelTree (const Path: String; const IsDir, DeleteFiles, DeleteSubdirsAlso: Boolean;
  const DeleteDirProc: TDeleteDirProc; const Param: Pointer): Boolean;
{ Deletes the specified directory including all files and subdirectories in
  it (including those with hidden, system, and read-only attributes). Returns
  True if it was able to successfully remove everything. }
var
  BasePath, FindSpec: String;
  H: THandle;
  FindData: TWin32FindData;
  S: String;
begin
  Result := True;
  if DeleteFiles then begin
    if IsDir then begin
      BasePath := AddBackslash(Path);
      FindSpec := BasePath + '*';
    end
    else begin
      BasePath := ExtractFilePath(Path);
      FindSpec := Path;
    end;
    H := FindFirstFile(PChar(FindSpec), FindData);
    if H <> INVALID_HANDLE_VALUE then begin
      repeat
        S := FindData.cFileName;
        if FindData.dwFileAttributes and FILE_ATTRIBUTE_READONLY <> 0 then
          SetFileAttributes (PChar(BasePath + S), FindData.dwFileAttributes and
            not FILE_ATTRIBUTE_READONLY);
        if FindData.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY = 0 then
          WinProcs.DeleteFile (PChar(BasePath + S))
        else begin
          if DeleteSubdirsAlso and (S <> '.') and (S <> '..') then
            if not DelTree(BasePath + S, True, True, True, DeleteDirProc, Param) then
              Result := False;
        end;
      until not FindNextFile(H, FindData);
      WinProcs.FindClose (H);
    end;
  end;
  if IsDir then begin
    if Assigned(DeleteDirProc) then begin
      if not DeleteDirProc(Path, Param) then
        Result := False;
    end
    else begin
      if not RemoveDirectory(PChar(Path)) then
        Result := False;
    end;
  end;
end;

function GetVersionInfo (const Filename: String;
  var VersionInfo: TVSFixedFileInfo): Boolean;
var
  VersionSize: Integer;
  VersionHandle: DWORD;
  VersionBuf: PChar;
  VerInfo: PVSFixedFileInfo;
  VerInfoSize: UINT;
begin
  Result := False;

  VersionSize := GetFileVersionInfoSize(StringAsPChar(Filename),
    VersionHandle);
  if VersionSize <> 0 then begin
    GetMem (VersionBuf, VersionSize);
    try
      if GetFileVersionInfo(StringAsPChar(Filename), VersionHandle, VersionSize, VersionBuf) then begin
        if VerQueryValue(VersionBuf, '\', Pointer(VerInfo), VerInfoSize) then begin
          VersionInfo := VerInfo^;
          Result := True;
        end;
      end;
    finally
      FreeMem (VersionBuf, VersionSize);
    end;
  end;
end;

function GetVersionNumbers (const Filename: String;
  var VersionNumbers: TFileVersionNumbers): Boolean;
var
  VerInfo: TVSFixedFileInfo;
begin
  Result := GetVersionInfo(Filename, VerInfo);
  if Result then begin
    VersionNumbers.MS := VerInfo.dwFileVersionMS;
    VersionNumbers.LS := VerInfo.dwFileVersionLS;
  end;
end;

procedure IncrementSharedCount (const Filename: String;
  const AlreadyExisted: Boolean);
const
  SharedDLLsKey = NEWREGSTR_PATH_SETUP + '\SharedDLLs';  {don't localize}
var
  K: HKEY;
  Disp, Size, Count, CurType, NewType: DWORD;
  CountStr: String;
  FilenameP: PChar;
begin
  if RegCreateKeyEx(HKEY_LOCAL_MACHINE, SharedDLLsKey, 0, nil, REG_OPTION_NON_VOLATILE,
     KEY_QUERY_VALUE or KEY_SET_VALUE, nil, K, @Disp) <> ERROR_SUCCESS then
    raise Exception.Create(FmtSetupMessage(msgErrorRegOpenKey,
      [RegRootKeyNames[HKEY_LOCAL_MACHINE], SharedDLLsKey]));
  FilenameP := PChar(Filename);
  Count := 0;
  NewType := REG_DWORD;
  try
    if RegQueryValueEx(K, FilenameP, nil, @CurType, nil, @Size) = ERROR_SUCCESS then
      case CurType of
        REG_SZ:
          if RegQueryStringValue(K, FilenameP, CountStr) then begin
            Count := StrToInt(CountStr);
            NewType := REG_SZ;
          end;
        REG_BINARY: begin
            if (Size >= 1) and (Size <= 4) then begin
              if RegQueryValueEx(K, FilenameP, nil, nil, @Count, @Size) <> ERROR_SUCCESS then
                { ^ relies on the high 3 bytes of Count being initialized to 0 }
                Abort;
              NewType := REG_BINARY;
            end;
          end;
        REG_DWORD: begin
            Size := SizeOf(DWORD);
            if RegQueryValueEx(K, FilenameP, nil, nil, @Count, @Size) <> ERROR_SUCCESS then
              Abort;
          end;
      end;
  except
    Count := 0;
  end;
  if Integer(Count) < 0 then Count := 0;  { just in case... }
  if (Count = 0) and AlreadyExisted then
    Inc (Count);
  Inc (Count);
  case NewType of
    REG_SZ: begin
        CountStr := IntToStr(Count);
        RegSetValueEx (K, FilenameP, 0, NewType, PChar(CountStr), Length(CountStr)+1);
      end;
    REG_BINARY, REG_DWORD:
      RegSetValueEx (K, FilenameP, 0, NewType, @Count, SizeOf(Count));
  end;
  RegCloseKey (K);
end;

function GetFileDateTime (const Filename: string; var DateTime: TFileTime): Boolean;
var
  Handle: THandle;
  FindData: TWin32FindData;
begin
  Handle := FindFirstFile(PChar(Filename), FindData);
  if Handle <> INVALID_HANDLE_VALUE then begin
    WinProcs.FindClose (Handle);
    if FindData.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY = 0 then begin
      DateTime := FindData.ftLastWriteTime;
      Result := True;
      Exit;
    end;
  end;
  Result := False;
  DateTime.dwLowDateTime := 0;
  DateTime.dwHighDateTime := 0;
end;

function GetFileLocalDateTime (const Filename: string; var DateTime: TFileTime): Boolean;
var
  DT: TFileTime;
begin
  Result := GetFileDateTime(Filename, DT);
  if Result then
    FileTimeToLocalFileTime (DT, DateTime);
end;

function InstExec (const Filename, Params: String; WorkingDir: String;
  const WaitUntilTerminated, WaitUntilIdle: Boolean; const ShowCmd: Integer;
  const ProcessMessagesProc: TProcedure; var ErrorCode: Integer): Boolean;
var
  CmdLine: String;
  WorkingDirP: PChar;
  StartupInfo: TStartupInfo;
  ProcessInfo: TProcessInformation;
begin
  Result := True;
  CmdLine := {$IFDEF WIN32}AddQuotes({$ENDIF}Filename{$IFDEF WIN32}){$ENDIF} +
    ' ' + Params {$IFNDEF WIN32} + #0 {needs null terminator for Delphi 1} {$ENDIF};
  if WorkingDir = '' then
    WorkingDir := RemoveBackslashUnlessRoot(ExtractFilePath(Filename));
  FillChar (StartupInfo, SizeOf(StartupInfo), 0);
  StartupInfo.cb := SizeOf(StartupInfo);
  StartupInfo.dwFlags := STARTF_USESHOWWINDOW;
  StartupInfo.wShowWindow := ShowCmd;
  if WorkingDir <> '' then
    WorkingDirP := PChar(WorkingDir)
  else
    WorkingDirP := nil;
  if not CreateProcess(nil, PChar(CmdLine), nil, nil, False, 0, nil,
     WorkingDirP, StartupInfo, ProcessInfo) then begin
    Result := False;
    ErrorCode := GetLastError;
    Exit;
  end;
  with ProcessInfo do begin
    { Don't need the thread handle, so close it now }
    CloseHandle (hThread);
    if WaitUntilIdle then
      WaitForInputIdle (hProcess, INFINITE);
    if WaitUntilTerminated then
      { Wait until the process returns, but still process any messages that
        arrive. }
      repeat
        { Process any pending messages first because MsgWaitForMultipleObjects
          (called below) only returns when *new* messages arrive }
        if Assigned(ProcessMessagesProc) then
          ProcessMessagesProc;
      until MsgWaitForMultipleObjects(1, hProcess, False, INFINITE, QS_ALLINPUT) <> WAIT_OBJECT_0+1;
    { Then close the process handle }
    CloseHandle (hProcess);
  end;
end;

function InstShellExec (const Filename, Params: String; WorkingDir: String;
  const ShowCmd: Integer; var ErrorCode: Integer): Boolean;
var
  WorkingDirP: PChar;
  E: Integer;
begin
  if WorkingDir = '' then
    WorkingDir := RemoveBackslashUnlessRoot(ExtractFilePath(Filename));
  if WorkingDir <> '' then
    WorkingDirP := PChar(WorkingDir)
  else
    WorkingDirP := nil;
  E := ShellExecute(0, 'open', PChar(Filename), PChar(Params), WorkingDirP,
    ShowCmd);
  Result := E > 32;
  if not Result then
    ErrorCode := E;
end;

function CheckForMutexes (Mutexes: String): Boolean;
{ Returns True if any of the mutexes in the comma-separated Mutexes string
  exist }
var
  I: Integer;
  M: String;
  H: THandle;
begin
  Result := False;
  repeat
    I := Pos(',', Mutexes);
    if I = 0 then I := Maxint;
    M := Trim(Copy(Mutexes, 1, I-1));
    if M <> '' then begin
      H := OpenMutex(SYNCHRONIZE, False, PChar(M));
      if H <> 0 then begin
        CloseHandle (H);
        Result := True;
        Break;
      end;
    end;
    Delete (Mutexes, 1, I);
  until Mutexes = '';
end;

function ModifyPifFile (const Filename: String; const CloseOnExit: Boolean): Boolean;
{ Changes the "Close on exit" setting of a .pif file. Returns True if it was
  able to make the change. }
var
  F: File;
  B: Byte;
begin
  { Note: Specs on the .pif format were taken from
    http://smsoft.chat.ru/en/pifdoc.htm }
  Result := False;
  AssignFile (F, Filename);
  FileMode := fmOpenReadWrite or fmShareExclusive;
  Reset (F, 1);
  try
    { Is it a valid .pif file? }
    if FileSize(F) >= $171 then begin
      Seek (F, $63);
      BlockRead (F, B, SizeOf(B));
      { Toggle the "Close on exit" bit }
      if (B and $10 <> 0) <> CloseOnExit then begin
        B := B xor $10;
        Seek (F, $63);
        BlockWrite (F, B, SizeOf(B));
      end;
      Result := True;
    end;
  finally
    CloseFile (F);
  end;
end;

function GetComputerNameString: String;
var
  Buf: array[0..MAX_COMPUTERNAME_LENGTH] of Char;
  Size: DWORD;
begin
  Size := SizeOf(Buf);
  if GetComputerName(Buf, Size) then
    Result := Buf
  else
    Result := '';
end;

function GetUserNameString: String;
var
  Buf: array[0..255] of Char;
  BufSize: DWORD;
begin
  BufSize := SizeOf(Buf);
  if GetUserName(Buf, BufSize) then
    Result := Buf
  else
    Result := '';
end;

{ Work around problem in D2's declaration of the function }
function NewAdjustTokenPrivileges(TokenHandle: THandle; DisableAllPrivileges: BOOL;
  const NewState: TTokenPrivileges; BufferLength: DWORD;
  PreviousState: PTokenPrivileges; ReturnLength: PDWORD): BOOL; stdcall;
  external advapi32 name 'AdjustTokenPrivileges';

procedure RestartComputer;
{ Restarts the computer. The function will NOT return if it is successful,
  since Windows kills the process immediately after sending it a WM_ENDSESSION
  message. }

  procedure RestartErrorMessage;
  begin
    MessageBox (0, PChar(SetupMessages[msgErrorRestartingComputer]),
      PChar(SetupMessages[msgErrorTitle]), MB_OK or MB_ICONEXCLAMATION);
  end;

var
  Token: THandle;
  TokenPriv: TTokenPrivileges;
const
  SE_SHUTDOWN_NAME = 'SeShutdownPrivilege';  { don't localize }
begin
  if Win32Platform = VER_PLATFORM_WIN32_NT then begin
    if not OpenProcessToken(GetCurrentProcess, TOKEN_ADJUST_PRIVILEGES or TOKEN_QUERY,
       {$IFNDEF Delphi3orHigher} @Token {$ELSE} Token {$ENDIF}) then begin
      RestartErrorMessage;
      Exit;
    end;

    LookupPrivilegeValue (nil, SE_SHUTDOWN_NAME, TokenPriv.Privileges[0].Luid);

    TokenPriv.PrivilegeCount := 1;
    TokenPriv.Privileges[0].Attributes := SE_PRIVILEGE_ENABLED;

    NewAdjustTokenPrivileges (Token, False, TokenPriv, 0, nil, nil);

    { Cannot test the return value of AdjustTokenPrivileges. }
    if GetLastError <> ERROR_SUCCESS then begin
      RestartErrorMessage;
      Exit;
    end;
  end;
  if not ExitWindowsEx(EWX_REBOOT, 0) then
    RestartErrorMessage;

  { If ExitWindows/ExitWindowsEx were successful, program execution halts here
    (at least on Win95) }
end;

procedure DelayDeleteFile (const Filename: String; const Tries: Integer);
{ Attempts to delete Filename, retrying up to Tries times if the file is in use.
  It delays 250 msec between tries. }
var
  I: Integer;
begin
  for I := 0 to Tries-1 do begin
    if I <> 0 then Sleep (250);
    if WinProcs.DeleteFile(PChar(Filename)) or
       (GetLastError = ERROR_FILE_NOT_FOUND) or
       (GetLastError = ERROR_PATH_NOT_FOUND) then
      Break;
  end;
end;


{ TSimpleStringList }

procedure TSimpleStringList.Add (const S: String);
begin
  if FCount = FCapacity then
    SetCapacity (FCapacity + 8);
  FList^[FCount] := S;
  Inc (FCount);
end;

procedure TSimpleStringList.AddIfDoesntExist (const S: String);
begin
  if IndexOf(S) = -1 then
    Add (S);
end;

procedure TSimpleStringList.SetCapacity (NewCapacity: Integer);
begin
  ReallocMem (FList, NewCapacity * SizeOf(Pointer));
  if NewCapacity > FCapacity then
    FillChar (FList^[FCapacity], (NewCapacity - FCapacity) * SizeOf(Pointer), 0);
  FCapacity := NewCapacity;
end;

procedure TSimpleStringList.Clear;
begin
  if FCount <> 0 then Finalize (FList^[0], FCount);
  FCount := 0;
  SetCapacity (0);
end;

function TSimpleStringList.Get (Index: Integer): String;
begin
  Result := FList^[Index];
end;

function TSimpleStringList.IndexOf (const S: String): Integer;
{ Note: This is case-sensitive, unlike TStringList.IndexOf }
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FCount-1 do
    if FList^[I] = S then begin
      Result := I;
      Break;
    end;
end;

destructor TSimpleStringList.Destroy;
begin
  Clear;
  inherited Destroy;
end;

end.
