unit Msgs;

{
  Inno Setup
  Copyright (C) 1998-2000 Jordan Russell
  For conditions of distribution and use, see LICENSE.TXT.

  Message file handling functions
}

interface

uses
  MsgIDs;

const
  SNewLine = #13#10;  { line break }
  SNewLine2 = #13#10#13#10;  { double line break }

var
  SetupMessageFile: Pointer;
  SetupMessageFileSize: Cardinal;
  SetupMessages: array[TSetupMessageID] of String;

function FmtMessage (S: PChar; const Args: array of String): String;
function FmtSetupMessage (const ID: TSetupMessageID; const Args: array of String): String;
function FmtSetupMessage1 (const ID: TSetupMessageID; const Arg1: String): String;
procedure LoadSetupMessages (const Filename: String; const Offset: Longint;
  const AFileMode: Byte);
procedure LoadCompressedSetupMessages (const Filename: String; const Offset: Longint;
  const AFileMode: Byte);
function IntToHexStr8 (I: Integer): String;
procedure StrFmtMessage (Buffer: PChar; MaxLen: Cardinal; S: PChar;
  const Args: array of String);
procedure StrFmtSetupMessage (Buffer: PChar; MaxLen: Cardinal;
  const ID: TSetupMessageID; const Args: array of String);
function AddPeriod (const S: String): String;
procedure ExtractFontNameSize (const ANameSize: String;
  var AName: String; var ASize: Integer);

const
  { You don't have to translate these. The only time they are used is when an
    error occurs before or while the messages file is loaded. Otherwise, it
    uses the corresponding messages in the messages file. }
  SSetupFileCorrupt = 'The setup files are corrupted. Please obtain a new ' +
    'copy of the program.';
  SMsgsFileMissing = 'Messages file "%s" is missing. Please correct ' +
    'the problem or obtain a new copy of the program.';

implementation

uses
  Windows, SysUtils, Struct, zlib;

const
  SMsgsFileTooLarge = 'Internal error: Messages file is too large';

procedure StrFmtMessage (Buffer: PChar; MaxLen: Cardinal; S: PChar;
  const Args: array of String);
var
  Buf: PChar;

  procedure Append (const S: String);
  var
    L: Cardinal;
  begin
    L := Length(S);
    if L > MaxLen then L := MaxLen;
    Buf := StrEnd(StrPLCopy(Buf, S, L));
    Dec (MaxLen, L);
  end;
var
  P: PChar;
  L: Cardinal;
begin
  Buf := Buffer;
  Buf[0] := #0;
  if S = nil then Exit;
  while MaxLen > 0 do begin
    P := StrScan(S, '%');
    if P = nil then begin
      StrLCopy (Buf, S, MaxLen);
      Break;
    end;
    if P <> S then begin
      L := P - S;
      if L > MaxLen then L := MaxLen;
      Buf := StrEnd(StrLCopy(Buf, S, L));
      Dec (MaxLen, L);
      if MaxLen = 0 then Break;
      S := P;
    end;
    Inc (P);
    if (P^ >= '1') and (Ord(P^) <= Ord('1') + High(Args)) then begin
      Append (Args[Ord(P^) - Ord('1')]);
      Inc (S, 2);
    end
    else begin
      Append ('%');
      Inc (S);
      if P^ = '%' then
        Inc (S);
    end;
  end;
end;

function FmtMessage (S: PChar; const Args: array of String): String;
var
  P: PChar;
  Z: String;
begin
  Result := '';
  if S = nil then Exit;
  while True do begin
    P := StrScan(S, '%');
    if P = nil then begin
      Result := Result + S;
      Break;
    end;
    if P <> S then begin
      SetString (Z, S, P - S);
      Result := Result + Z;
      S := P;
    end;
    Inc (P);
    if (P^ >= '1') and (Ord(P^) <= Ord('1') + High(Args)) then begin
      Result := Result + Args[Ord(P^) - Ord('1')];
      Inc (S, 2);
    end
    else begin
      Result := Result + '%';
      Inc (S);
      if P^ = '%' then
        Inc (S);
    end;
  end;
end;

procedure StrFmtSetupMessage (Buffer: PChar; MaxLen: Cardinal;
  const ID: TSetupMessageID; const Args: array of String);
begin
  StrFmtMessage (Buffer, MaxLen, PChar(SetupMessages[ID]), Args);
end;

function FmtSetupMessage (const ID: TSetupMessageID; const Args: array of String): String;
begin
  Result := FmtMessage(PChar(SetupMessages[ID]), Args);
end;

function FmtSetupMessage1 (const ID: TSetupMessageID; const Arg1: String): String;
begin
  Result := FmtSetupMessage(ID, [Arg1]);
end;

procedure FreeSetupMessages;
var
  I: TSetupMessageID;
begin
  for I := Low(SetupMessages) to High(SetupMessages) do
    SetupMessages[I] := '';
  if Assigned(SetupMessageFile) then
    FreeMem (SetupMessageFile, SetupMessageFileSize);
  SetupMessageFile := nil;
  SetupMessageFileSize := 0;
end;

procedure LoadSetupMessages (const Filename: String; const Offset: Longint;
  const AFileMode: Byte);
  procedure Corrupted;
  begin
    raise Exception.Create(SSetupFileCorrupt);
  end;
type
  PMessageLengths = ^TMessageLengths;
  TMessageLengths = array[TSetupMessageID] of Smallint;
var
  F: File;
  TestID: TMessagesHdrID;
  Header: TMessagesHeader;
  I: TSetupMessageID;
  Lengths: PMessageLengths;
  M: PChar;
  P: Pointer;
  CRC: Longint;
begin
  FreeSetupMessages;
  if not FileExists(Filename) then
    raise Exception.CreateFmt(SMsgsFileMissing, [Filename]);
  AssignFile (F, Filename);
  FileMode := AFileMode;  Reset (F, 1);
  try
    if Offset + SizeOf(TestID) + SizeOf(Header) > FileSize(F) then
      Corrupted;
    Seek (F, Offset);
    BlockRead (F, TestID, SizeOf(TestID));
    if TestID <> MessagesHdrID then
      Corrupted;
    BlockRead (F, Header, SizeOf(Header));
    if (Header.TotalSize <> not Header.NotTotalSize) or
       (Header.NumMessages <> (Ord(High(SetupMessages)) - Ord(Low(SetupMessages)) + 1)) then
      Corrupted;
    if Offset + Header.TotalSize > FileSize(F) then
      Corrupted;

    Seek (F, Offset);
    GetMem (P, Header.TotalSize);
    try
      BlockRead (F, P^, Header.TotalSize);
      Longint(Lengths) := Longint(P) + SizeOf(TestID) + SizeOf(Header);
      if GetCRC32(Lengths^, SizeOf(Lengths^)) <> Header.CRCLengths then
        Corrupted;
      Longint(M) := Longint(Lengths) + SizeOf(TMessageLengths) + Header.Padding;
      CRC := Longint($FFFFFFFF);
      for I := Low(SetupMessages) to High(SetupMessages) do begin
        SetupMessages[I] := M;
        CRC := UpdateCRC32(CRC, M^, Lengths^[I]);
        Inc (M, Lengths^[I]);
      end;
      if (not CRC) <> Header.CRCMessages then
        Corrupted;
    except
      FreeMem (P, Header.TotalSize);
      FreeSetupMessages;
      raise;
    end;
    SetupMessageFile := P;
    SetupMessageFileSize := Header.TotalSize;
  finally
    CloseFile (F);
  end;
end;

procedure LoadCompressedSetupMessages (const Filename: String; const Offset: Longint;
  const AFileMode: Byte);
  procedure Corrupted;
  begin
    raise Exception.Create(SSetupFileCorrupt);
  end;
type
  PMessageLengths = ^TMessageLengths;
  TMessageLengths = array[TSetupMessageID] of Smallint;
var
  F: File;
  Data: PDeflateBlockReadData;
  TestID: TMessagesHdrID;
  Header: TMessagesHeader;
  I: TSetupMessageID;
  Lengths: PMessageLengths;
  M: PChar;
  P: Pointer;
  CRC: Longint;
begin
  FreeSetupMessages;
  if not FileExists(Filename) then
    raise Exception.CreateFmt(SMsgsFileMissing, [Filename]);
  AssignFile (F, Filename);
  FileMode := AFileMode;  Reset (F, 1);
  Data := nil;
  try
    New (Data);
    try
      Seek (F, Offset);
      InflateBlockReadBegin (F, Data^);
      try
        InflateBlockRead (Data^, TestID, SizeOf(TestID));
        if TestID <> MessagesHdrID then
          Corrupted;
        InflateBlockRead (Data^, Header, SizeOf(Header));
        if (Header.TotalSize <> not Header.NotTotalSize) or
           (Header.NumMessages <> (Ord(High(SetupMessages)) - Ord(Low(SetupMessages)) + 1)) then
          Corrupted;
      finally
        InflateBlockReadEnd (Data^);
      end;
      Seek (F, Offset);
      InflateBlockReadBegin (F, Data^);
      try
        GetMem (P, Header.TotalSize);
        try
          InflateBlockRead (Data^, P^, Header.TotalSize);
          Longint(Lengths) := Longint(P) + SizeOf(TestID) + SizeOf(Header);
          if GetCRC32(Lengths^, SizeOf(Lengths^)) <> Header.CRCLengths then
            Corrupted;
          Longint(M) := Longint(Lengths) + SizeOf(TMessageLengths) + Header.Padding;
          CRC := Longint($FFFFFFFF);
          for I := Low(SetupMessages) to High(SetupMessages) do begin
            SetupMessages[I] := M;
            CRC := UpdateCRC32(CRC, M^, Lengths^[I]);
            Inc (M, Lengths^[I]);
          end;
          if (not CRC) <> Header.CRCMessages then
            Corrupted;
        except
          FreeMem (P, Header.TotalSize);
          FreeSetupMessages;
          raise;
        end;
        SetupMessageFile := P;
        SetupMessageFileSize := Header.TotalSize;
      finally
        InflateBlockReadEnd (Data^);
      end;
    except
      on EZlibDataError do
        Corrupted;
    end;
  finally
    CloseFile (F);
    if Assigned(Data) then
      Dispose (Data);
  end;
end;

function IntToHexStr8 (I: Integer): String;
begin
  FmtStr (Result, '0x%.8x', [I]);
end;

function AddPeriod (const S: String): String;
begin
  Result := S;
  if (Result <> '') and (Result[Length(Result)] > '.') then
    Result := Result + '.';
end;

function FontExistsCallback (const lplf: TLogFont; const lptm: TTextMetric;
  dwType: DWORD; lpData: LPARAM): Integer; stdcall;
begin
  Boolean(Pointer(lpData)^) := True;
  Result := 1;
end;

function FontExists (const FaceName: String): Boolean;
var
  DC: HDC;
begin
  Result := False;
  DC := GetDC(0);
  try
    EnumFonts (DC, PChar(FaceName), @FontExistsCallback, @Result);
  finally
    ReleaseDC (0, DC);
  end;
end;

procedure ExtractFontNameSize (const ANameSize: String;
  var AName: String; var ASize: Integer);
{ Note: AName and ASize are not set if ANameSize does not contain a name or
  size. Therefore, they should be initialized before calling this function. }
var
  P, S, E: Integer;
  ValidSize: Boolean;
  N: String;
begin
  P := Pos(',', ANameSize);
  if P = 0 then begin
    N := ANameSize;
    S := 0;  { avoid compiler warning }
    ValidSize := False;
  end
  else begin
    N := Copy(ANameSize, 1, P-1);
    Val (Trim(Copy(ANameSize, P+1, Maxint)), S, E);
    ValidSize := E = 0;
  end;
  N := Trim(N);
  if (N <> '') and FontExists(N) then
    AName := N;
  if ValidSize then
    ASize := S;
end;

initialization
finalization
  FreeSetupMessages;
end.
