/*
 * Copyright (c) 2018-2024, Arm Limited and Contributors. All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <common/debug.h>
#include <drivers/arm/gic600_multichip.h>
#include <plat/arm/common/plat_arm.h>
#include <plat/common/platform.h>

#include <nrd_plat.h>

#define RT_OWNER 0

#if defined(IMAGE_BL31)
static const mmap_region_t rdn1edge_dynamic_mmap[] = {
	NRD_CSS_SHARED_RAM_MMAP(1),
	NRD_CSS_PERIPH_MMAP(1),
	NRD_ROS_PERIPH_MMAP(1)
};

static struct gic600_multichip_data rdn1e1_multichip_data __init = {
	.base_addrs = {
		PLAT_ARM_GICD_BASE
	},
	.rt_owner = RT_OWNER,
	.chip_count = NRD_CHIP_COUNT,
	.chip_addrs = {
		[RT_OWNER] = {
			PLAT_ARM_GICD_BASE >> 16,
			(PLAT_ARM_GICD_BASE
				+ NRD_REMOTE_CHIP_MEM_OFFSET(1)) >> 16
		}
	},
	.spi_ids = {
		{PLAT_ARM_GICD_BASE,
		NRD_CHIP0_SPI_MIN,
		NRD_CHIP0_SPI_MAX},
		{0, 0, 0}
	}
};

static uintptr_t rdn1e1_multichip_gicr_frames[] = {
	PLAT_ARM_GICR_BASE,				/* Chip 0's GICR Base */
	PLAT_ARM_GICR_BASE +
		NRD_REMOTE_CHIP_MEM_OFFSET(1),		/* Chip 1's GICR BASE */
	UL(0)						/* Zero Termination */
};
#endif /* IMAGE_BL31 */

unsigned int plat_arm_nrd_get_platform_id(void)
{
	return mmio_read_32(SID_REG_BASE + SID_SYSTEM_ID_OFFSET)
				& SID_SYSTEM_ID_PART_NUM_MASK;
}

unsigned int plat_arm_nrd_get_config_id(void)
{
	return mmio_read_32(SID_REG_BASE + SID_SYSTEM_CFG_OFFSET);
}

unsigned int plat_arm_nrd_get_multi_chip_mode(void)
{
	return (mmio_read_32(SID_REG_BASE + SID_NODE_ID_OFFSET) &
			SID_MULTI_CHIP_MODE_MASK) >> SID_MULTI_CHIP_MODE_SHIFT;
}

/*
 * IMAGE_BL31 macro is added to build bl31_platform_setup function only for BL31
 * because PLAT_XLAT_TABLES_DYNAMIC macro is set to build only for BL31 and not
 * for other stages.
 */
#if defined(IMAGE_BL31)
void bl31_platform_setup(void)
{
	unsigned int i;
	int ret;

	if (plat_arm_nrd_get_multi_chip_mode() == 0 && NRD_CHIP_COUNT > 1) {
		ERROR("Chip Count is set to %d but multi-chip mode not enabled\n",
				NRD_CHIP_COUNT);
		panic();
	} else if (plat_arm_nrd_get_multi_chip_mode() == 1 &&
			NRD_CHIP_COUNT > 1) {
		INFO("Enabling support for multi-chip in RD-N1-Edge\n");

		for (i = 0; i < ARRAY_SIZE(rdn1edge_dynamic_mmap); i++) {
			ret = mmap_add_dynamic_region(
					rdn1edge_dynamic_mmap[i].base_pa,
					rdn1edge_dynamic_mmap[i].base_va,
					rdn1edge_dynamic_mmap[i].size,
					rdn1edge_dynamic_mmap[i].attr
					);
			if (ret != 0) {
				ERROR("Failed to add dynamic mmap entry\n");
				panic();
			}
		}

		plat_arm_override_gicr_frames(rdn1e1_multichip_gicr_frames);
		gic600_multichip_init(&rdn1e1_multichip_data);
	}

	nrd_bl31_common_platform_setup();
}
#endif /* IMAGE_BL31 */
