package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.36.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.36.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,``P`!````D"(``#0````PP0```````#0`(``,
M`"@`&P`:``8````T````-````#0```"``0``@`$```0````$`````P```+0!
M``"T`0``M`$``!,````3````!`````$````!````````````````````K!$`
M`*P1```$`````!````$`````(````"`````@``"4,```E#````4`````$```
M`0````!@````8````&```"!.```@3@``!``````0```!````V+T``-B]``#8
MO0``0`(``%0"```&`````!````(```#@O0``X+T``."]```(`0``"`$```8`
M```$````!````,@!``#(`0``R`$``&P```!L````!`````0```!3Y71D[`$`
M`.P!``#L`0``*````"@````$````!````%#E=&1,H```3*```$R@```4`0``
M%`$```0````$````4>5T9```````````````````````````!@```!````!2
MY71DV+T``-B]``#8O0``*`(``"@"```$`````0```"]L:6(O;&0M;&EN=7@N
M<V\N,@``!````!0````#````1TY5`#]&D0BHTDJ@:Y%52Q;9KO-TVDL+!```
M`!@````%````1TY5``$``<`$`````0````(``<`$`````0````0````0````
M`0```$=.50``````!`````0`````````+P```$0````(````"``````0$`0E
M0`2`*`$`!:"&`(2``0``"R@$"#B`$D&*```(``````````!$````````````
M``!%````````````````````1P```$@`````````20````````!*````````
M`$L```````````````````!,`````````$T``````````````$\```!0````
M`````%$``````````````%,`````````5````%4``````````````%8```!8
M````60```%H```!;````7````````````````````%T```!K"4/6+L6;MF_%
MWQ^S`8MO0_K7AVM_FGRM2^/`6>CM[.-'Q@$,.I<+P^4_+"'_Y`Y_J9R;X*<"
M"KOCDGS59-1\S]Y'+?NA;&!J&IY0H^#[23-BV^T5C"7,9Z_,9D-%U>R)!W#3
MV7%8'`````````````````````"M`@`````````````2````%P(`````````
M````$@```!$"`````````````!(````_`0`````````````2````@```````
M````````$@```+```````````````!(```#3`0`````````````2````/0(`
M````````````$@```'P!`````````````!(````!```````````````@````
M)P(`````````````$@```(("`````````````!(```#0`@`````````````2
M````IP``````````````$@```%,``````````````!(```"[````````````
M```2````V@$`````````````$@````4"`````````````!(````0````````
M```````@````'P$`````````````$@```$0!`````````````!(```"8`@``
M```````````2````+```````````````(````$P"`````````````!(```")
M`@`````````````2````CP$`````````````$@````P!`````````````!8`
M``"7```````````````2````;P``````````````$@```&L!````````````
M`!(```#8```````````````2````N0$`````````````$@```&4"````````
M`````!$```"?`@`````````````2````^`$`````````````$@```.(!````
M`````````!(```"/`P`````````````2````R0``````````````$@```)$"
M`````````````!(```#(`@`````````````1````R@$`````````````$@``
M`.<``````````````!(````*`@`````````````2````50$`````````````
M$@```/4``````````````!(```!C```````````````2````X@(`````````
M````$@```/\!`````````````!(```#Q`0`````````````2````2P(`````
M````````$@```!X"`````````````!(```#I`0`````````````2````*@$`
M````````````$@```$D!`````````````!(````O`@`````````````2````
MQ@(`````````````$0```&P"`````````````!(```"^`@`````````````2
M````1@``````````````$0```(P``````````````!(```!<`0``````````
M```2````PP$`````````````$@```%,"`````````````!(```"Q`0``````
M```````2````I@(`````````````$@```',"`````````````"(```"L`0``
M```````````2````I`,```#`````````$``7``@#``!@+```!0```!(`#@"8
M`P``,#\``-$````2``X`0P,``&`T``!A````$@`.`+$#``!03```QP$``!(`
M#@!@`@``4"```#P"```2``X`G0$```1@```$````$0`0`"8#````8```!```
M`!$`$``9`P``H"X``#8````2``X`I@,```#`````````(``7`+P#``!@*P``
M-````!(`#@`$!```<"P``"<"```2``X`?0,``-`T```M````$@`.`%$#``"@
M*P``O````!(`#@#9`P``+,`````````0`!@`+0,``"`]```!`@``$@`.`-X#
M````-@``'@<``!(`#@#R`P```#4``/D````2``X`<P,``"!.```0`@``$@`.
M`!L$``!@,```_`,``!(`#@"J`P``D"(``#`````2``X`T`,```!,``!0````
M$@`.`&<#``#@+@``=`$``!(`#@#K`P``&,`````````0`!<`$00``!3````$
M````$0`7`,0#```8P````````!``&```7U]G;6]N7W-T87)T7U\`7TE435]D
M97)E9VES=&5R5$U#;&]N951A8FQE`%])5$U?<F5G:7-T97)434-L;VYE5&%B
M;&4`4$Q?9&]?=6YD=6UP`%!E<FQ?9W9?9F5T8VAP=@!097)L7VUG7W-E=`!0
M97)L7V%T9F]R:U]L;V-K`%!E<FQ?9V5T7W-V`%!E<FQ?;F5W6%,`8F]O=%]$
M>6YA3&]A9&5R`'!E<FQ?<G5N`'!E<FQ?<&%R<V4`4&5R;%]S=E]S971P=@!0
M97)L7W-V7W-E='!V;@!P97)L7V-O;G-T<G5C=`!P97)L7V1E<W1R=6-T`%!E
M<FQ?8W)O86M?;65M;W)Y7W=R87``4$Q?8W5R<F5N=%]C;VYT97AT`'!E<FQ?
M86QL;V,`4&5R;%]C<F]A:U]N;V-O;G1E>'0`<&5R;%]F<F5E`%!E<FQ?<V%F
M97-Y<VUA;&QO8P!097)L7W-Y<U]I;FET,P!097)L7VYE=UA37V9L86=S`%!E
M<FQ?871F;W)K7W5N;&]C:P!097)L7W-Y<U]T97)M`%])3U]S=&1I;E]U<V5D
M`&5X:70`;'-E96LV-`!R96%D9&ER-C0`<W1R9'5P`&-L;W-E9&ER`&]P96XV
M-`!S=')R8VAR`'5N;&EN:P!M96UM;W9E`'-T<G1O:P!S=')L96X`<FUD:7(`
M<F5A9`!S=')S='(`;6MD:7(`9V5T<&ED`')E861L:6YK`')E86QL;V,`7U]S
M<')I;G1F7V-H:P!?7V9P<FEN=&9?8VAK`&QS=&%T-C0`7U]L:6)C7W-T87)T
M7VUA:6X`<W1D97)R`&UE;6-H<@!?7V-X85]F:6YA;&EZ90!S=')C:'(`;W!E
M;F1I<@!G971U:60`86-C97-S`&UE;6-P>0!S=')C;7``7U]E<G)N;U]L;V-A
M=&EO;@!S=')N8VUP`%]?96YV:7)O;@!?7W)E9VES=&5R7V%T9F]R:P!?7W-T
M86-K7V-H:U]F86EL`&QI8G!E<FPN<V\`;&EB8RYS;RXV`'!A<E]C=7)R96YT
M7V5X96,`<&%R7V)A<V5N86UE`%]F<%]H=P!84U]);G1E<FYA;'-?4$%27T)/
M3U0`<&%R7V5N=E]C;&5A;@!P87)?8W5R<F5N=%]E>&5C7W!R;V,`<&%R7V1I
M<FYA;64`<VAA7V9I;F%L`&=E=%]U<V5R;F%M95]F<F]M7V=E='!W=6ED`'!A
M<E]C;&5A;G5P`%]?9&%T85]S=&%R=`!S:&%?=7!D871E`'AS7VEN:70`7U]B
M<W-?<W1A<G0`<VAA7VEN:70`7V5N9`!P87)?;6MT;7!D:7(`7V5D871A`'!A
M<E]S971U<%]L:6)P871H`'!A<E]F:6YD<')O9P!M>5]P87)?<&P`<&%R7VEN
M:71?96YV`$=,24)#7S(N,2XS`$=,24)#7S(N,S0`1TQ)0D-?,BXT`$=,24)#
M7S(N,@!'3$E"0U\R+C,S`$=,24)#7S(N,RXR`$=,24)#7S(N,RXT`$=,24)#
M7S(N,0!'3$E"0U\R+C``+W5S<B]L:6(O<&5R;#4O-2XS-B]C;W)E7W!E<FPO
M0T]210`````"``(``@`!``$``0`#``0``0`!``(``@`%``$``0`!``(``@`!
M``$``@`"``$`!@`"``$``0`!``$``0`!``<``@`"``(``@`"``$``@`"``(`
M`0`"``(``0`!``@``@`"``8``@`"``$``0`$``(``@`"``$``0`!``(`"0`#
M``(`"@`"``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$`
M`0`!``$``0`!``$``0````$`"0#^`@``$`````````!S'VD)```*`"@$```0
M````M)&6!@``"0`T!```$````!1I:0T```@`/P0``!`````2:6D-```'`$D$
M```0````LY&6!@``!@!3!```$````'(9:0D```4`7@0``!````!T&6D)```$
M`&H$```0````$6EI#0```P!V!```$````!!I:0T```(`@`0```````#8O0``
M"````-R]```(````_+\```@````$P```"````!3````(````^+X```8!``#\
MO@``!@(```"_```&`P``!+\```8$```(OP``!@4```R_```&!@``$+\```8'
M```4OP``!@@``!B_```&"0``'+\```8*```@OP``!@L``"2_```&#```*+\`
M``8-```LOP``!@X``#"_```&#P``-+\```80```XOP``!A$``#R_```&$@``
M0+\```83``!$OP``!A0``$B_```&%0``3+\```86``!0OP``!A<``%2_```&
M&```6+\```89``!<OP``!AH``&"_```.&P``9+\```8<``!HOP``!AT``&R_
M```&'@``<+\```8?``!TOP``!B```'B_```&(0``?+\```8B``"`OP``!B,`
M`(2_```&)```B+\```8E``",OP``!B8``)"_```&)P``E+\```8H``"8OP``
M!BD``)R_```&*@``H+\```8K``"DOP``!BP``*B_```&+0``K+\```8N``"P
MOP``!B\``+2_```&,```N+\```8Q``"\OP``!C(``,"_```&,P``Q+\```8T
M``#(OP``!C4``,R_```&-@``T+\```8W``#4OP``!CD``-B_```&.@``W+\`
M``8[``#@OP``!CP``.2_```&/0``Z+\```8^``#LOP``!D```/"_```&00``
M]+\```9"``#XOP``!D,``/2^```'/P``````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``#S#Q[[4X/L".BS`@``@</;G@``BX,T````A<!T`O_0@\0(6\,`````````
M``````#_LP0```#_HP@`````````_Z,,````:`````#IX/___XU,)`2#Y/#_
M<?Q5B>57B<]64^A9`@``@<.!G@``48/L+(M!!(E%U(M!"(E%T&6A%````(E%
MY#'`C4704(U%U%!1_Y/\````BX,P````@\0,4%#_LX````!GZ((O``"+@_0`
M``"#Q!"`.``/A"4!``"+@W@```!EBP"`B)H"```#BP>-<`.#P`2)=<P]____
M/P^'A@$``(/L#,'@`E#_D^0```"+==2+/XG!B8,\`0``BP:#Q!")`8V#WZ+_
M_XE!!(N#+`$``(E!"(V#XJ+__XE!#(/_`0^.1P$``(E-R(U&!(U1$(TTOHVT
M)@````!FD(L(@\`$@\($B4K\.<9U\8M-R(/'`X/L#(V#&&S__\<$N0````!J
M`%'_=<Q0_[-``0``_Y,D````B<:#Q""%P`^$E@```(/L#/^S0`$``(U]X/^3
MM````(GZC8._H?__Z!P"``"#Q!"%P`^$A0```(/L#/^S0`$``/^3'````/^3
M=````(/$$(M%Y&4K!10````/A;````"-9?")\%E;7E]=C6'\P_^37````(F#
M0`$``(7`#X2#````@^P,4/^3B````(N#>````&6+$(/$$,:"0P8```#IJ_[_
M_X/L#/^S0`$``/^31````(G&@\00Z5'___^+@S@!``"%P'01@^P,4.CI'```
M@\00Z6#___^)^HV#=*'__^AD`0``B8,X`0``A<`/A$7____KTO^3P````+\$
M````Z=G^__^#[`QJ`?^3$`$``.C4+0``9I!FD/,/'OLQ[5Z)X8/D\%!44N@8
M````@<-$G```:@!J`%%6_[,4`0``Z(7]___TBQPDPXL<),-FD&:09I!FD&:0
M9I#HY````('"$YP``(V*,`$``(V",`$``#G(=!V+@E@```"%P'0358GE@^P4
M4?_0@\00R<.-="8`D,.-M"8`````Z*0```"!PM.;``!5B>53C8HP`0``C8(P
M`0``@^P$*<B)P\'H'\'[`@'8T?AT%(N2:````(72=`J#[`A04?_2@\00BUW\
MR<.-M"8`````9I#S#Q[[58GE4^A3____@<-[FP``@^P$@+LP`0```'4HBX,,
M`0``A<!T$H/L#/^S'`$``/^3#`$``(/$$.@P____QH,P`0```8M=_,G#C70F
M`/,/'OOI5____XL4),-FD)!5Z$<<``"!Q2*;``!75E.#[!R)5"0(A<`/A+(`
M``")P8N%K````(L0A=(/A*`````/M@&$P'0$/#UU53'_BQJ%VW0]B50D#(G8
MB=.)3"0$B<:-M"8`````@^P$5_]T)`Q6_Y7P````@\00A<!U!H`\/CUT-(MS
M!(/#!(7V==J)\X/$'(G86UY?7<.-M@````")SP^V1P&#QP&$P'0$/#UU\2G/
MZYB-=@"+5"0,B?")WHG#BT0D""G6C5P[`<'^`HDP@\0<B=A;7E]=PXUT)@"0
M@\0<,=N)V%M>7UW#C70F`%;H7QL``('&0IH``%.)PX/L?&6A%````(E$)'0Q
MP(U$)!104_^6U````(/$$(7`=3>+5"0<@>(`\```@?H`0```=`B!^@"@``!U
M'X/L"&H"4_^69````(/$$(7`#Y3`#[;`ZP:-="8`,<"+5"1L92L5%````'4&
M@\1T6U[#Z#XK``"-M"8`````C;0F`````%575NC-&@``@<:PF0``4XG#@>R,
M````9:$4````B40D?#'`A=MT!8`[`'4DBT0D?&4K!10````/A3P!``"!Q(P`
M``!;7E]=PXVT)@````"0@^P,4_^6<````(G'@\00A<!TR8/L#%?_EHP```"#
MQ!"%P`^$N````(!X$RZ-4!-U#X!Z`0!TW(VT)@````!FD(!X$RX/A+8```")
M5"0,@^P,4_^6F````(M4)!R)Q8D4)(E4)!S_EI@```"-1`4"B00D_Y:\````
MB<58C88@H?__6HM4)!124U!J_VH!5?^6Z````(/$&(U$)"105?^6;````(/$
M$(/X_W00BT0D+"4`\```/0!```!T98/L#%7_EIP```"#Q!"#[`Q5_Y9@````
M@\00Z3G___^-M@````"#[`Q7_Y:P````B1PD_Y;,````@\00Z>'^__^-="8`
MD(!Z`2X/A4#___^`>@(`#X3__O__Z3'___^-M"8`````B>CHB?[__^N?Z+(I
M``!FD%57B<=64^@%_/__@<,MF```@>PL``$`B40D!&6A%````(F$)!P``0`Q
MP&H":@!J`%?_DP0!``"#P/^#TO^)1"08BW0D&(G7B50D','_'S'X*?@QTB7_
M_P``,?HQ^"GX&?J+?"0<*<89UX/$$(UL)!2%_P^(I````(VT)@````"0:@!7
M5O]T)!#_DP0!``"#Q`QH"``!`%7_="00_Y-4````@\00@_@'=E>-1`7X.>AR
M3XT5(`$``.L.C;0F`````(/H`3GH<CF+#!HY"'7RBXLD`0``.4@$=><IZ)D!
M\!'ZBXPD'``!`&4K#10```!U.('$+``!`%M>7UW#C70F`)"!Q@``__^#U_^)
M\(GZ-0``___WT@G0#X5D____N/____^Z_____^NXZ(,H``"-=@!5Z"<8``"!
MQ0*7``!7B<=64XG3@^PX9:$4````B40D*#'`,<"`.CT/E,`!PU/_E9@```"#
MQ!")1"0$B?B-5"08Z)O[__^%P`^$$P$``(/L#(G&4/^5F````(M,)!2#Q!`Y
MR`^#V````(M$)!C!X`*)1"0(BX6L````B00D#[8'A,`/A*D!```\/0^$H0$`
M`(GXC;0F``````^V4`&#P`&$TG0%@/H]=?`I^(L,)(MT)`B#[`R+5"00`S&-
M1!`"4/^5O````(D&@\00A<!T48L,)(MT)`B+`8U/`8L4,`^V!X@"/#UT'83`
M=0KK%XUT)@`\/70/#[8!@\(!@\$!B`*$P'7MQ@(],<"-M"8`````D`^V#`.(
M3`(!@\`!A,EU\8M$)!QE*P44````#X43`0``@\0L6UY?7<.-M"8`````D#'`
MC;8`````#[84`X@4!H/``832=?+KQXVT)@````"0BX6L````BQ")!"2+,H7V
M#X3"````,?:-M"8`````9I")\(/&`8L,LH7)=?2-!(4,````BXTT`0``A<EU
M8XE4)`R#[`Q0QX4T`0```0```/^5O````(G!@\00A<`/A%W___^-!+4`````
M@^P$B40D#%"+5"044E&)3"0<_Y7<````BT0D$(M,)!R#Q!")",=$L00`````
MB70D&.E]_O__D(/L"%!2_Y4X````B<&+1"00@\00B0B%R0^$`?___XT$M0``
M``")1"0(Z\.-="8`D#'`Z7/^__^X"````#'VZ5#____H."8``(VT)@````"0
M55?HTA4``(''L90``%:)QE.![#`0``")5"009:$4````B80D(!```#'`:/\/
M``"-AR:A__^-;"0D55#_E]@```"#Q!"%P'X0B<.-0/^`?!P;`'5$A<!U.(N'
M>````(/L!/]T)!!69?\P_Y=,````@\00BX0D'!```&4K!10```!U08'$+!``
M`%M>7UW#C78`B<.-M@````"#[`13:B]5_Y?L````@\00A<!TK(N'>````%-5
M5F7_,/^7I````(/$$.NOZ&@E``"-M"8`````D%93Z+GW__^!P^&3``"#[`2-
M@UCA__^+LW@```!0C8,UH?___[-\````4&7_-O^3^````%B-@[RB__]::@!0
M4(V#.'[__U"-@U&A__]09?\V_Y.$````@\0D6U[#C70F`.B<%```!8.3``"#
M[`R-D%CA__]2C9`UH?___[!\````4HN0>````&7_,O^0^````(/$',.-M"8`
M````C70F`)!5,>U75E/H%??__X'#/9,``('L'"```&6A%````(F$)`P@```Q
MP/^3%````(/L"(V3,:'__U)0C8-FH?__4&@!$```:@&-="0F5O^3Z````(/$
M'&@`$```C;PD$Q```%=6_Y/8````@\00A<!X+(/L#(G&C4`!4/^3O````(G%
M@\00A<!T%(/L!%974/^3E````,9$-0``@\00BX0D#"```&4K!10```!U#8'$
M'"```(GH6UY?7</H!"0``(UT)@#I.____XVT)@````"-="8`55=64^A']O__
M@<-OD@``@>R<$```9:$4````B80DC!```#'`C40D+(N\)+`0``"+K"2T$```
MB40D%(G"C8-TH?__Z`GW__^#[`AJ+XG&5_^3/````(/$$(7`=#2-@WVA__^)
M^N@'^___B?B+E"2,$```92L5%`````^%HP$``('$G!```%M>7UW#C;8`````
M@^P,5?^3``$``%I9C8N*H?__B4PD%%%0_Y/0````C8N,H?__B<6-@WRB__^#
MQ!")1"08A>UTE8!]``")3"0<#X3J````C78`A?9T%H/L"%56_Y,(`0``@\00
MA<`/A*<```"#[`Q5_Y.8````B<*-1`7_@\00.<5R$>LCC70F`)#&``"#Z`$Y
MQ70%@#@O=/&#[`Q5_Y.8````B<*#Q!")5"00@^P,5_^3F````(M4)""-1`(!
M@\00/?\/```/APK___^#[`17_W0D)%7_="0H:``0``!J`8VL)*@```!5_Y/H
M````@\08_W0D'%7_DVP```"#Q!"%P'40BT0D/"4`\```/0"```!T5X/L"/]T
M)!1J`/^3T````(G%@\00A<`/A*?^__^`?0``#X49____A?9T!8`^+G08N@$`
M``"-JW*A___I5O___XVT)@````"0@'X!`'2SZ^"-M"8`````D(/L"&H!5?^3
M9````(/$$(7`=9:)ZHV#?:'__^A@^?__@^P,5?^3``$``(/$$.E)_O__Z,DA
M``"-M"8`````9I!6Z%\1``"!QD*0``!3@^P,BUPD&&HO4_^64````(7`C5`!
MB?`/1=J)7"0@@\046U[_H``!``"-M"8`````C78`5>@G$0``@<4"D```5U93
M@>P<$```9:$4````B80D#!```#'`B[0D,!```(7V#X2M````@#X`#X2D````
M@^P,5O^5F````(U(`8/$$('Y`!````^'Z````(U<)`R#^00/@ZL```"%R0^%
MXP```(U$`_\YPW(5ZQB-M"8`````9I#&``"#Z`$YV'0%@#@O=/&#[`AJ+U/_
ME5````"#Q!"%P'0^.<,/@H8```"#[`Q3_Y4``0``@\00BY0D#!```&4K%10`
M```/A:0```"!Q!P0``!;7E]=PXVT)@````"-=@"+A"0,$```92L%%````'5_
MC85RH?__B80D,!```('$'!```(GH6UY?7?^@``$``)"+5`[\B=^)5`P(B<'!
MZ0+SI>E'____C;0F`````(UV`,8``.ER____C;0F`````)`QP.EP____C;0F
M`````&:0#[86B%0D#/;!`@^$#?___P^W5`[^9HE4#`KI_O[__^@,(```C;0F
M`````(UT)@"055=6Z)T/``"!QH".``!3@^PL9:$4````B40D'#'`C5PD&(NN
MK````(V^CJ'__XG:B?CH*?/__X7`="Z+5"08BT4`C020BU`$B1"%TG3@9I"+
M4`B#P`2)$(72=?2)VHGXZ/OR__^%P'72BZZL````C;Z7H?__B=J)^.CB\O__
MA<!T+XM4)!B+10"-!)"+4`2)$(72=."-=@"+4`B#P`2)$(72=?2)VHGXZ+/R
M__^%P'71BZZL````C;Z?H?__B=J)^.B:\O__A<!T+XM4)!B+10"-!)"+4`2)
M$(72=."-=@"+4`B#P`2)$(72=?2)VHGXZ&OR__^%P'71BZZL````C;ZHH?__
MB=J)^.A2\O__A<!T+XM4)!B+10"-!)"+4`2)$(72=."-=@"+4`B#P`2)$(72
M=?2)VHGXZ"/R__^%P'71C8:OH?__B[ZL````B00DBP0DB=KH!O+__X7`="R+
M5"08BP>-!)"+4`2)$(72=."+4`B#P`2)$(72=?2+!"2)VNC:\?__A<!UU(NN
MK````(V^OZ'__XG:B?CHP?'__X7`=#:+5"08BT4`C020BU`$B1"%TG3@C;0F
M`````(UV`(M0"(/`!(D0A=)U](G:B?CHB_'__X7`=<J-AG2A__^+OJP```")
M1"0$BT0D!(G:Z&SQ__^%P'0SBU0D&(L'C020BU`$B1"%TG3?C;8`````BU`(
M@\`$B1"%TG7TBT0D!(G:Z#GQ__^%P'7-C8;+H?__B[ZL````B40D#(M$)`R)
MVN@:\?__A<!T,8M4)!B+!XT$D(M0!(D0A=)TWXUT)@"+4`B#P`2)$(72=?2+
M1"0,B=KHZ?#__X7`=<^-AM6A__^+OJP```")1"0(BT0D"(G:Z,KP__^%P'0Q
MBU0D&(L'C020BU`$B1"%TG3?C70F`(M0"(/`!(D0A=)U](M$)`B)VNB9\/__
MA<!USXNNK````(V^WZ'__XG:B?CH@/#__X7`=#6+5"08BT4`C020BU`$B1"%
MTG3@C;0F`````&:0BU`(@\`$B1"%TG7TB=J)^.A+\/__A<!URXNNK````(V^
M?:'__XG:B?CH,O#__X7`="^+5"08BT4`C020BU`$B1"%TG3@C78`BU`(@\`$
MB1"%TG7TB=J)^.@#\/__A<!UT8G:C8;IH?__Z/+O__^%P'0+B<*+1"0(Z`/T
M__^)VHV&^J'__^C6[___A<!T#8G"C88,HO__Z.7S__^)VHV&%Z+__^BX[___
MB<*%P'0KBT0D!.C)\___BT0D'&4K!10```!U,XL$)(/$+(V6.*+__UM>7UWI
MI_/__XG:C88GHO__Z'KO__^%P'3-B<*+1"0,Z(OS___KP.@$'```C70F`.B<
M"P``!8.*``"#[!QEBQ44````B50D##'2C50D"(V`RZ'__^@W[___A<!T*P^V
M$(#Z,`^4P(32#Y3""="#\`$/ML"+5"0,92L5%````'4,@\0<PXUT)@`QP.OG
MZ)\;``"-M"8`````C;0F`````)!3Z.KM__^!PQ**``"#[`C_DZ@```"#[`Q0
M_Y.@````@\00A<!T`HL`@\0(6\.-=@!55U;H_0H``('&X(D``%.#["QEH10`
M``")1"0<,<"-5"08BVPD0(V^.J+__XGXZ(WN__^%P'0'@#@`B<-U(HM$)!QE
M*P44````#X6E````B>J#Q"R)^%M>7UWI@?+__Y"#[`A54/^6N````(/$$(7`
M=!Z+1"0<92L%%````'5U@\0L6UY?7<.-M"8`````9I"#[`Q5_Y:8````B40D
M'(D<)/^6F````(M4)!R-1`("B00D_Y:\````@\0,B<*-AHJA__]34(V&?*+_
M_U50:O]J`5*)5"0L_Y;H````@\0@BU0D#(M$)!QE*P44````#X1=____Z&<:
M``"-M"8`````55=64^BW[/__@</?B```@>S\````BX0D$`$``(U\)"R)^HE$
M)`QEH10```")A"3L````,<"-@U&B___'A"24`````````(F$)(P```"-@W*A
M__^)A"20````C8,%HO__B80DJ````(V#5J+__XF$)*P```"-@R*B__^)A"2P
M````C8->HO__B80DM````(V#8J+__XF$))@```"-@V>B__^)A"2<````C8-T
MH?__QX0DN`````````#'A"2@`````````(E$)!CH\^S__X7`=`N`.`")Q@^%
MC`0``/^3J````(/L#%#_DZ````"#Q!"%P`^$V0,``(LHA>T/A,\#``"#[`Q5
M_Y.8````C40``8/$$(/L#%#_D[P```")Q@^V10"#Q!")="0(A,!T.XV+<*+_
M_XE\)`2)]XG.C70F`)"#[`P/ML"#Q0%05FK_:@%7@\<"_Y/H````#[9%`(/$
M((3`==R+?"0$C;0DI````(V##*+__^L3C;0F`````&:0BT8$@\8$A<!T-HGZ
MZ"_L__^)Q87`=.F`.`!TY.C_[/__A<!TVX/L#%7_DP`!``")Q8/$$(7`=3B-
MM"8`````D(VT)(P```"++H7M=".`?0``#X3Q`@``B>CHPNS__X7`#X7R````
MBVX$@\8$A>UUW8/L#%7_DY@```!:_W0D%(G&_Y.8````C;0&!`0``(DT)/^3
MO````(E$)!2)P8V#=:+___]T)!A0C8.,H?__B40D(%"-@WJB__]5B<U0:O]J
M`5'_D^@```"#Q"AHP`$``%7_DQ@```"#Q!"#^/]U$?^3$````(L`@_@1#X78
M`@``@^P(5_]T)!#_D]0```")Q8/$$(7`='F#[`S_="00BT0D'/\PC8,DH___
M4#'VBX.0````:@'_,/^3+````(/$((N$).P```!E*P44````#X55!```@<3\
M````B?!;7E]=PXVT)@````"#[`R#Q@15_Y,``0``B<6#Q!"%P`^$U?[__^GY
M_O__C;0F`````&:0BT0D/"4`\```/0!````/A7/___^+1"1$B40D$/^3J```
M`#E$)!`/A5O___^+1"0\)?\!```]P`$```^%1____X/L#%;_D[P```")^HG&
MC8-%HO__Z&?J__^#Q!"%P`^$1`(``(/L"%"+1"08_S#H_?+__XE$)""#Q!"%
MP`^$)@(``(/L#/]T)!S_DY@```")PH/$$(/X`W9'@^P(C8.#HO__4(M$)!R-
M1!#\B50D*%#_DP@!``"#Q!"%P'4CBU0D'(/Z!`^$]@(``(M$)!"`?!#[+P^$
MYP(``(VT)@````")^HV[RZ'__XGXZ-'I__^%P`^$R0$```^V`(3`#X2^`0``
M/#`/A+8!``"-DSBB__^)^(V[O*+__^C#[?___Y,4````5U"-@YRB____="00
M_W0D$%!J_VH!5O^3Z````(V#JJ+__X/$((E$)`SK-F:0_Y,0````@S@1=4#_
MDQ0```"#Q0&#[`Q755#_="0@_W0D(/]T)"QJ_VH!5O^3Z````(/$,(/L"&C`
M`0``5O^3&````(/$$(/X_W2U@^P,_W0D$/^38````(M$)"B)\N@M[?__B30D
MZ$7Z__^#Q!#I[?W__XUT)@"0,>WI(?W__Y"+A"28````C;0DF````(7`=""-
MM@````")^NC1Z/__A<!T!8`X`'4XBT8$@\8$A<!UYH/L#(VK2J+__VH-_Y.\
M````B<:#Q!")1"0(N%,```#I!_S__XVT)@````"-=@"#[`Q0_Y,``0``B<6#
MQ!"%P'2UZ;G[__^#[`A0_W0D$(M$)!S_,(V#Z*+__^DY_?__C;0F`````(/L
M#%#HC_G__X/$$(N$).P```!E*P44````#X61`0``B;0D$`$``('$_````(G8
M6UY?7?^@``$``(VT)@````"+1"0,BP")1"00Z<O]__^-M"8`````9I"#[`AJ
M`/]T)!S_DR@```"#Q!"%P`^$,/[__XE$)`SHS>K__XE$)!")5"04A=(/B!?^
M__^#P/9J`(/2_U)0BTPD&%&)3"0<_Y,$`0``@\0,:@:-A"3%````4(M,)!A1
MB4PD'/^35````(/$$(M,)`R!O"2]`````$-!0P^%ROW__V:!O"3!````2$4/
MA;K]__]J`(M$)!2+5"08@\#.@]+_4E!1B4PD'/^3!`$``(/$#&HHC;PDRP``
M`%>+3"084?^35````(V#O*+__\:$)/L`````4(V#C:+__U?_="0@_W0D(%!J
M_VH!5O^3Z````(/$,.GD_?__C;0F`````&:0BT0D#(M0!(72#X01_?__@^P,
M4HE4)!S_DY@```"#Q!"#^`,/AO?\__^#[`B-BXBB__]1BU0D&(U$`OQ0_Y,(
M`0``BU0D'(/$$(7`#T54)!")5"00Z<?\___H0A,``&:055=64^B7Y?__@<._
M@0``@^P<9:$4````B40D##'`C7PD"(GZB[-X````98LN#[9%9(A%98V#KZ'_
M_^AAYO__A<!T"8`X,0^$5`$``.CO\O__98LNC8.[HO__:@-J`5!5_Y-(````
MB<6#Q!"%P'0J@^P$C8.]HO__4(M%#/\P9?\V_Y-,````BT4,BP"#Q!#V0`I`
M#X4W`0``C8/"HO__:@-J`5!E_S;_DT@```")Q8/$$(7`="J)^HV#?:'__^C=
MY?__A<`/A.4```"#[`10BT4,_S!E_S;_DTP```"#Q!"#[`2-@\2B__]J`5!E
M_S;_DR````"-D]FB__^#Q`Q24&7_-O^33````&6+!HGZQD!E`(V#=*'__^B#
MY?__@\00B8,X`0``B<:%P'1$@^P(:,`!``!6_Y,8````C5`!@\00@_H!=@F#
M^!$/A98```"+1"0,92L%%`````^%G````(/$'%M>7UW#C;0F`````)"#[`S_
MLSP!``#H8O?__X/$$(F#.`$``(G&A<!UGX/L#(V#>*/__U#_D^````"-=@"`
M>`$`#X2J_O__Z9W^__^0BX,\`0``BQ"+10R+`.A.Z___Z1G___^-M"8`````
M9I"#[`A09?\V_Y/$````@\00Z;3^__]0BX,\`0``5O\PC8.<H___4/^3X```
M`.@_$0``C;0F`````(VT)@````"05U;HS@```('&L7\``%.#[!QEH10```")
M1"08,<"+?"0L5^B)[___6EEJ+XG#4/^64````(7`C5`!#T7:B1PD_Y8``0``
MC50D&(G#C8;+H?__Z#KD__^#Q!"%P'0C#[80A-(/E,"`^C`/E,((T'41A?]T
M#8`_`'4@C;0F`````)"+1"0,92L%%````'4_@\006UY?PXUT)@"#[`2-AG6B
M__]J!%!3_Y;P````@\00A<!USHM$)`QE*P44````=0V#Q!")^%M>7^DTY?__
MZ%\0``"+!"3#BS0DPXL\),.++"3#9I!FD&:09I!FD&:09I"055=64XG#@>Q\
M`0``B40D$(U,)"QEH10```")A"1L`0``,<"-="8`D(M4@QP/RHD4@8/``8/X
M$'7OBTPD,(MT)&"-5"0TBVPD9(M$)&B)3"0(C8PD-`$``(M<)"R)3"0,ZPJ-
MM@````")3"0(BPJ)QXM"&(/"!#'(,?")[HG],=B+7"0(T<")0C0Y5"0,==F+
M?"00BT0D+(LWBU<0BV\(BU\,B50D'(V4`IEY@EJ)\(M/!,'`!8GWB5PD"`'"
MB>B)3"04,=B)?"0,(<C!R0*);"08,=B+7"0P`<*)Z#'((?"+="0(,>B-M!Z9
M>8):BUPD-`'&B=#!P`6-G!V9>8):BVPD.`'&B?C!R`*)QXG(,?@AT,'*`C'(
MC8PIF7F"6HML)#P!PXGPP<`%`<.)^#'0(?#!S@(Q^(V\+YEY@EJ+;"1``<B)
MV<'!!0'(B=$Q\2'9,=&-E"J9>8):BVPD1`'YB<?!QP7!RP(!^8GW,=\AQ\'(
M`C'WC;0NF7F"6HML)$@!^HG/P<<%`?J)QXG8,?@AR#'8C9PKF7F"6HML)$P!
M\(G6P<8%`?")SHGYP<X",?$AT3'YC;POF7F"6HML)%`!V8G#P<,%`=F)TXGR
MP<L",=HAPL'(`C'RC;0NF7F"6HML)%0!^HG/P<<%`?J)QXG8,?@AR#'8C9PK
MF7F"6HML)%@!\(G6P<8%`?")SHGYP<X",?$AT3'YC;POF7F"6HML)%P!V8G#
MP<,%`=F)TXGRP<L",=HAPL'(`C'RC;0NF7F"6HML)&`!^HG/P<<%`?J)QXG8
M,?@AR#'8C9PKF7F"6HML)&0!\(G6P<8%`?")SHGYP<X",?$AT3'YC;POF7F"
M6@'9B</!PP4!V8G3B?+!RP(QVB'"P<@",?*)Q8G8`?J)SS'HP<<%(<C!R0(!
M^HM\)&@QV(VT/IEY@EJ+?"1L`?")UL'&!8V\.YEY@EH!\(GN,<Z)PR'6P<,%
M,>[!R@(!]XMT)'`!WXG+,=.-M#69>8):BVPD="'#P<@",<N-C"F9>8):BVPD
M>`'>B?O!PP4!WHG3,<,A^S'3P<\"C90JF7F"6@'+B?'!P04!RXG!,?DA\<'.
M`C'!`=&)VL'"!0'1BU0D?(V$$*'KV6Z)^C'R,=K!RP(!PHG(P<`%`<*+A"2`
M````C;P'H>O9;HGP,=@QR,')`@'XB=?!QP4!^(G/BXPDA````(VT#J'KV6Z)
MV3'Y,=$!\8G&P<8%`?&)UHN4)(@```#!S@*-G!.AZ]ENB?HQ\C'"`=J)R\'(
M`L'#!0':B<.+A"2,````C;P'H>O9;HGP,=@QR,')`@'XB=?!QP4!^(G/BXPD
MD````(VT#J'KV6Z)V3'Y,=$!\8G&P<8%`?&)UHN4))0```#!S@*-G!.AZ]EN
MB?HQ\C'"P<@"`=J)R\'#!0':B<.+A"28````C;P'H>O9;HGP,=@QR,')`@'X
MB=?!QP4!^(G/BXPDG````(VT#J'KV6Z)V3'Y,=$!\8G&P<8%`?&)UHN4)*``
M``#!S@*-G!.AZ]ENB?HQ\C'"P<@"`=J)R\'#!0':B<.+A"2D````C;P'H>O9
M;HGP,=@QR,')`@'XB=?!QP4!^(G/BXPDJ````(VT#J'KV6Z)V3'Y,=$!\8G&
MP<8%`?&)UHN4)*P```#!S@*-G!.AZ]ENB?HQ\C'"P<@"`=J)R\'#!0':B<.+
MA"2P````C;P'H>O9;HGP,=@QR,')`@'XB=?!QP4!^(G/BXPDM````(VT#J'K
MV6Z)V3'Y,=$!\8G&P<8%`?&)UHN4)+@```#!S@*-G!.AZ]ENB?HQ\C'"P<@"
M`=J)R\'#!0':B<.+A"2\````C;P'H>O9;HGP,=@QR,')`@'XB=?!QP4!^(G/
MBXPDP````(VT#J'KV6Z)V3'Y,=$!\8G&P<8%`?&)UHN4),0```#!S@*-G!.A
MZ]ENB?HQ\C'"P<@"B<4!VHN$),@```")R\'#!0':C9P'H>O9;HGP,>@QR`'8
MB=/!R0+!PP4!V(GK"<N)WXGK(=<AR\'*`@G[B[PDS````(VT/MR\&X\!\XG&
MP<8%`?.)S@G6B?>)SB''(=;!R`()_HN\)-````"-O#W<O!N/`?>)WL'&!0'W
MB=8)QHGUB=8AW2'&P<L"">Z+K"34````C8PIW+P;CXG=B<,!SHGY">O!P04A
M^P'.B<$AZ0G9BYPDV````(V<&MR\&X^)\@'+P<\"B>G!P@4)^0'3B>HA\<'.
M`B'Z"<J+C"3<````C8P(W+P;CXG8`='!P`4!P8GX"?")PHGX(=HA\,'+`@G0
MBY0DX````(V4%=R\&X\!PHG(P<`%`<*)\`G8B<6)\"'-(=C!R0()Z(NL).0`
M``"-O"_<O!N/B=T!^(G7"<W!QP4AU0'XB<^)V2'Y">F+K"3H````C;0NW+P;
MCXG]`?&)QL'&!0'QB=:)^L'.`@GU(?(AQ<'(`@GJBZPD[````(V<*]R\&X\!
MVHG+P<,%`=J)PXGP"=B)Q8GP(<TAV,')`@GHBZPD\````(V\+]R\&X^)W0'X
MB=<)S<''!2'5`?B)SXG9(?D)Z8NL)/0```"-M"[<O!N/B?T!\8G&P<8%`?&)
MUHGZP<X""?4A\B'%">J+K"3X````C9PKW+P;CP':B<O!R`+!PP4!VHG#B?`)
MV(G%B?`AS2'8P<D"">B+K"3\````C;POW+P;CXG=`?B)UPG-P<<%(=4!^(G/
MB=DA^0GIBZPD``$``(VT+MR\&X^)_0'QB<;!Q@4!\8G6B?K!S@()]2'R(<7!
MR`()ZHNL)`0!``"-G"O<O!N/B<4!VHG+P<,%`=J)\PG#B?`ARR'H"=B+G"0(
M`0``C9P?W+P;CP'8B=/!R0+!PP4!V(GK"<N)WXGK(=<AR\'*`@G[B[PD#`$`
M`(VT/MR\&X\!\XG&P<8%`?.)S@G6B?>)SB''(=;!R`()_HN\)!`!``"-O#W<
MO!N/B=4!_HG?"<7!QP4AW<'+`@'^B=<AQPGOBZPD%`$``(V,*=R\&X^)Q0'Y
MB?<)W<''!2'U`?F)QR'?">^+K"08`0``C90JW+P;CXML)`P!UXG*P<X"P<(%
M`=>+E"0<`0``C900UL%BRHG8,?`QR,')`@'0B?K!P@4!T(N4)"`!``"-G!/6
MP6+*B?(QRC'ZP<\"`=J)P\'#!0':BYPD)`$``(VT'M;!8LJ)RS'[,<,!\XG6
MP<8%`?.)QHN$)"@!``#!S@*-C`'6P6+*B?@Q\#'0P<H"`<B)V<'!!0'(B=&+
ME"0L`0``C907UL%BRHGW,<\QWP'7B<+!P@4!UXN4)#`!``#!RP*-M!;6P6+*
MB<HQVC'"`?*)_L'&!0'RB<:+A"0T`0``P<X"C80!UL%BRHG9,?$Q^<'/`@'!
MB=#!P`4!P8N$)#@!``"-G`/6P6+*B?`Q^#'0`=B)R\'#!0'8B=.+E"0\`0``
MP<L"C906UL%BRHG^,=XQSL')`@'6B<+!P@4!UHN4)$`!``"-O!?6P6+*B=HQ
MRC'"P<@"`?J)]\''!0'ZB<>+A"1$`0``C80#UL%BRHG+,?LQ\\'.`@'#B=#!
MP`4!PXN$)$@!``"-C`'6P6+*B?@Q\#'0P<H"`<B)V<'!!0'(B=&+E"1,`0``
MC907UL%BRHGW,<\QW\'+`@'7B<+!P@4!UXN4)%`!``"-M!;6P6+*B<HQVC'"
M`?*)_L'&!0'RB<:+A"14`0``P<X"C80!UL%BRHG9,?$Q^<'/`@'!B=#!P`4!
MP8N$)%@!``"-G`/6P6+*B?`Q^#'0`=B)R\'#!0'8B=.+E"1<`0``P<L"C906
MUL%BRHG^,=XQSL')`@'6B<+!P@4!UHN4)&`!``"-O!?6P6+*B=HQRC'"P<@"
M`?J)]\''!0'ZB[PD9`$``(V\.];!8LJ)RS'#,?/!S@(!^XG7P<<%`?N+O"1H
M`0``C;P]UL%BR@'/B<$Q\3'1P<H"`?F)W\''!0'YBWPD$(D/BTPD%`'9B4\$
MBVPD&(M,)`@!ZHE7"(T4,8E7#(M4)!P!T(E'$(N$)&P!``!E*P44````=0N!
MQ'P!``!;7E]=P^AA!```D.C\\___!>-R``"#[!AJ8/^0O````,<``2-%9\=`
M!(FKS>_'0`C^W+J8QT`,=E0R$,=`$/#ATL/'0!0`````QT`8`````,=`7```
M``"#Q!S#55=6Z*WS__^!QI!R``!3@^P<BUPD,(M$)#B+?"0TP>`#`T,4BTL8
MC6L<B4,4BT0D.(/1`,'H'0'(B4,8BT-<A<`/A3$!``"#?"0X/P^.;@$``(M$
M)#B#Z$")1"0,P>@&C7`!B40D",'F!@'^C70F`(L'@\=`B44`BT?$B44$BT?(
MB44(BT?,B44,BT?0B440BT?4B444BT?8B448BT?<B44<BT?@B44@BT?DB44D
MBT?HB44HBT?LB44LBT?PB44PBT?TB44TBT?XB44XBT?\B44\B=CH]_+__SGW
M=9.+1"0(BU0D#,'@!BG"B50D.(-\)#@$<E.+!HGIB4,<BT0D.(M4)#B+1`;\
MB405_(U#((/@_"G!*<X#3"0X@^'\@_D$<A*#X?PQTHLL%HDL$(/"!#G*<O.+
M1"0XB4-<@\0<6UY?7<.-="8`D(M$)#B%P'3D#[8&B$,<]D0D.`)TUXM$)#B+
M3"0X#[=$!OYFB40-_NO#C;0F`````+I`````*<([5"0X#T]4)#B#[`0!Z%*)
M5"005U#_EI0```"+5"08BT-<`="#Q!")0UR#^$!UC8G8*50D.`'7Z!7R___I
MA_[__XG^Z2?___^-M"8`````9I#HW/'__P7#<```55=64X/L'(M<)#2+="0P
MB40D!(M3%(MK&(U['(G0P>H##\V#XC\/R(E$)`B-0@'&1!,<@(/X.`^."0$`
M`+I`````C0P'*<)T(S'`]L(!=`RX`0```,8!`#G0<Q#&!`$`QD0!`0"#P`(Y
MT'+PB=CH@_'__XU7!,=#'``````QP(/B_,='-``````IUXU/.(G7P>D"\ZN+
M1"0(B6M4B4-8B=CH4/'__XL#P>@8B`:+`\'H$(A&`8L#B&8"BP.(1@,/MD,'
MB$8$#[=#!HA&!8M#!(AF!HM#!(A&!P^V0PN(1@@/MT,*B$8)BT,(B&8*BT,(
MB$8+#[9##XA&#`^W0PZ(1@V+0PR(9@Z+0PR(1@\/MD,3B$80#[=#$HA&$8M#
M$(AF$HM#$(A&$XM$)`2)7"0P@\0<6UY?7?^@8````(VT)@````"-=@"Z.```
M`"G"`?B#^@0/@HX```"-2`3'``````"#X?S'1!#\`````"G(`="#X/R#^`0/
M@AG___^)Q[H$````@^?\B7PD#(UX_\'O`L<!`````(/G`3G"#X/T_O__A?]T
M&,=!!`````"+1"0,N@@````YP@^#V/[__XM$)`S'!!$`````QT01!`````"#
MP@@YPG+JZ;G^__^-M"8`````C78`A=(/A*?^___&``#VP@(/A)O^__\QR6:)
M3!#^Z8_^___S#Q[[Z,CO__\%KVX``(/L#/^P'`$``/]T)!S_="0<_W0D'/^0
M0````(/$',-FD)#S#Q[[5E[HEN___P5];@``@^P,_Y#(````````\P\>^U.#
M[`CH-]+__X'#7VX``(/$"%O#````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````P````$``@`E<R\E<P`O<')O8R]S96QF+V5X90!$>6YA3&]A9&5R
M.CIB;V]T7T1Y;F%,;V%D97(`26YT97)N86QS.CI005(Z.D)/3U0`+W!R;V,O
M)6DO)7,`+@!005)?5$5-4`!005)?4%)/1TY!344`.@`O`%!%4DPU3$E"`%!%
M4DQ,24(`4$523#5/4%0`4$523$E/`%!!4E])3DE424%,25I%1`!005)?4U!!
M5TY%1`!005)?0TQ%04X`4$%27T1%0E5'`%!!4E]#04-(10!005)?1TQ/0D%,
M7T1%0E5'`%!!4E]'3$]"04Q?5$U01$E2`%!!4E]435!$25(`4$%27T=,3T)!
M3%]414U0`%!!4E]'3$]"04Q?0TQ%04X`,0!,1%],24)205)97U!!5$@`4UE3
M5$5-`"]T;7``5$5-4$1)4@!435``55-%4@!54T523D%-10`E,#)X`'!A<BT`
M)7,E<R5S)7,`<&%R;``N<&%R`"5S)7-C86-H92TE<R5S`"5S)7-T96UP+25U
M)7,`)7,E<W1E;7`M)74M)74E<P`8`'!E<FP`,`!005(Z.E!A8VME<CHZ5D52
M4TE/3@`Q+C`U.``M90`M+0`````E<SH@8W)E871I;VX@;V8@<')I=F%T92!S
M=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*```E<SH@<')I=F%T
M92!S=6)D:7)E8W1O<GD@)7,@:7,@=6YS869E("AP;&5A<V4@<F5M;W9E(&ET
M(&%N9"!R971R>2!Y;W5R(&]P97)A=&EO;BD*``!5;F%B;&4@=&\@8W)E871E
M(&-A8VAE(&1I<F5C=&]R>0`````E<SH@8W)E871I;VX@;V8@<')I=F%T92!C
M86-H92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*`````&5V
M86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII9B`P
M.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?7W!A
M<E]P;#L*;7D@*"1005)?34%'24,L("1&24Q%7V]F9G-E=%]S:7IE+"`D8V%C
M:&5?;F%M95]S:7IE*3L*0D5'24X@>PHD4$%27TU!1TE#(#T@(EQN4$%2+G!M
M7&XB.PHD1DE,15]O9F9S971?<VEZ92`](#0[("`@(R!P86-K*").(BD*)&-A
M8VAE7VYA;65?<VEZ92`](#0P.PHD4$M:25!?34%'24,@/2`B4$M<,#`S7#`P
M-"(["GT*<W5B(&9I;F1?<&%R7VUA9VEC"GL*;7D@*"1F:"D@/2!`7SL*;7D@
M)&-H=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY("1B=68["FUY("1S:7IE(#T@
M+7,@)&9H.PIM>2`D<&]S(#T@*"1S:7IE+3$I("T@*"1S:7IE+3$I("4@)&-H
M=6YK7W-I>F4[("`@("`@(R!.3U1%.B`D<&]S(&ES(&$@;75L=&EP;&4@;V8@
M)&-H=6YK7W-I>F4*=VAI;&4@*"1P;W,@/CT@,"D@>PIS965K("1F:"P@)'!O
M<RP@,#L*<F5A9"`D9F@L("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H*"10
M05)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L("1005)?34%'
M24,I*2`^/2`P*2!["G)E='5R;B`D<&]S("L@)&D["GT*)'!O<R`M/2`D8VAU
M;FM?<VEZ93L*?0IR971U<FX@+3$["GT*;7D@*"1P87)?=&5M<"P@)'!R;V=N
M86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%3E9[
M4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL
M93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I<B`]
M($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H<7%;
M4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN9#HZ
M9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL("1P
M87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M<"`F
M)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ5&5M
M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A<V5N
M86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I53DQ)
M3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64["G!R
M:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@
M9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@
M73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N
M90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P,"P@
M)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL(#(^
M)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG<F]U
M;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI.PI]
M"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F:6YE
M9"`F26YT97)N86QS.CI005(Z.D)/3U0["F5V86P@>PI?<&%R7VEN:71?96YV
M*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY("5S>7,@/2`H
M"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G.B<I
M+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I
M;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^("]>
M35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE*"D[
M"F]U=',H<7%;7"1P<F]G;F%M92`]("(D<')O9VYA;64B72D["E]S971?<&%R
M7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM005)?
M5$5-4'TB72D["FUY("@D<W1A<G1?<&]S+"`D<W1A<G1?;V9?1DE,15]S96-T
M:6]N*3L*34%'24,Z('L*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@>WT[
M"G5N;&5S<R`H;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G;F%M92D@>PIO=71S
M*'%Q6T-A;B=T(')E860@9G)O;2!F:6QE("(D<')O9VYA;64B72D[("`C(&1O
M;B=T('5S92`D(2!H97)E(&%S(&ET(')E<75I<F5S($5R<FYO+G!M"FQA<W0@
M34%'24,["GT*;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"I?1D@I
M.PII9B`H)&UA9VEC7W!O<R`\(#`I('L*;W5T<RAQ<5M#86XG="!F:6YD(&UA
M9VEC('-T<FEN9R`B)%!!4E]-04=)0R(@:6X@9FEL92`B)'!R;V=N86UE(ETI
M.PIL87-T($U!1TE#.PI]"F]U=',H(D9O=6YD(%!!4B!M86=I8R!A="!P;W-I
M=&EO;B`D;6%G:6-?<&]S(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)
M3$5?;V9F<V5T7W-I>F4@+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*<F5A9"!?
M1D@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I.PII9B`H)&)U9B!N92`B7#!#
M04-(12(@)B8@)&)U9B!N92`B7#!#3$5!3B(I('L*;W5T<R@B3F\@8V%C:&4@
M;6%R:V5R(&9O=6YD(BD["FQA<W0@34%'24,["GT*;W5T<RAQ<5M#86-H92!M
M87)K97(@(B1B=68B(&9O=6YD72D["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@
M)$9)3$5?;V9F<V5T7W-I>F4L(#`["G)E860@7T9(+"`D8G5F+"`D1DE,15]O
M9F9S971?<VEZ93L*;7D@)&]F9G-E="`]('5N<&%C:R@B3B(L("1B=68I.PIO
M=71S*")/9F9S970@9G)O;2!S=&%R="!O9B!&24Q%<R!I<R`D;V9F<V5T(BD[
M"G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2`D
M;V9F<V5T+"`P.PHD<W1A<G1?;V9?1DE,15]S96-T:6]N(#T@=&5L;"!?1D@[
M"FUY("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@("`@
M("`@("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B"G=H
M:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A
M9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA;64@
M/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B+BXN
M72D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO
M?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L("1E>'0I(#T@*"1B
M=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B=68L
M(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@*&1E
M9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I
M*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B=68L
M(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@/2`D
M9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE?2`]
M("1F=6QL;F%M93L*?0IE;'-E('L*)$UO9'5L94-A8VAE>R1F=6QL;F%M97T@
M/2!["F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L;&YA
M;64L"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK.PI]"G)E860@7T9(
M+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S96QF+"`D
M;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U;&4@;W(@(21M;V1U
M;&4["FUY("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U;&5].PIR971U<FX@
M=6YL97-S("1I;F9O.PID96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE?3L*
M)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B.PII9B`H
M)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W*2D@
M>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE(")#86XG
M="!C<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I.PHD9F@M
M/G!R:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*<F5T=7)N
M("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1I;F9O
M+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O+3Y[9FEL97T@/2`D
M9FEL96YA;64["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA;64@;W(@
M9&EE('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A73L*<F5T=7)N("1F
M:#L*?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86XG="!F:6YD(&UO
M9'5L92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R.PIR
M97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I
M<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A
M9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA;64@
M/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H)$E.
M0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII9B`H
M)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA;64[
M"GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^("]S
M:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI<W1[
M)&9I;&5N86UE?3L*?0II9B`H)&)U9B!N92`D4$M:25!?34%'24,I('L*;W5T
M<RAQ<5M.;R!Z:7`@9F]U;F0@869T97(@1DE,12!S96-T:6]N(&EN(&9I;&4@
M(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0R`["GT*)'-T87)T7W!O<R`]("AT
M96QL(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T87)T(&]F('II<`I]
M"FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@)&-A
M8VAE7VYA;64I.PID96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S86YI
M=&EZ92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*:68@*"$D
M<W1A<G1?<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@)B8@
M<VAI9G0I*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P87(*
M:2`@(&EN<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N7W!A
M<@IV("`@=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=(&5Q
M("<M+7)E=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%55-%
M?2`]('-H:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E:&%V
M:6]U<@IM>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=66S!=
M(#U^("]>+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF("@D
M,2!E<2`G22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@*"0Q
M(&5Q("=-)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G02<I
M('L*=6YS:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G3R<I
M('L*)&]U="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L92`]
M("=S:71E)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G86QL
M)SL*?0IE;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS:68@
M*"0Q(&5Q("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE('%Q
M6T-A;B=T(&]P96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D,2!E
M<2`G5"<I('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD["FEF
M("AM>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG4$%2
M7U1%35`G?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[(E!!
M4CHZ1&ES=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I<W0Z
M.B1C;60B?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`24Y#
M+"!`861D7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)3SHZ
M1FEL93L*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI3
M2$$["GT*;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@*&1E
M9FEN960@)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O<B!D
M:64@<7%;0V%N)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z("0A73L*8FQE<W,H
M)&9H+"`G24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH
M("1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R8VAI
M=F4Z.EII<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II
M<"!A<F-H:79E("(D<&%R(ET["GT*;7D@)6UE=&%?<&%R(#T@9&\@>PII9B`H
M)'II<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT<R@G345402YY;6PG
M*2D@>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>7%,N
M*B\O;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I].PII
M9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G/#IR87<G+"`D<&%R
M(&]R(&1I92!Q<5M#86XG="!R96%D('!A<B!F:6QE("(D<&%R(CH@)"%=.PIM
M>2`D8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I<R!N
M;W0@82!P87(@9FEL95T@=6YL97-S("1B=68@97$@)%!+6DE07TU!1TE#.PIC
M;&]S92`D<&@["GT*0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]($E/.CI&
M:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ1FEL
M93HZ3U]21%=2*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI(&]R
M(&1I92!Q<5M#86XG="!C<F5A=&4@9FEL92`B)&]U="(Z("0A73L*)&9H+3YB
M:6YM;V1E*"D["G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H9&5F
M:6YE9"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*2!["G)E860@7T9(+"`D;&]A
M9&5R+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N.PI](&5L<V4@>PIL;V-A;"`D
M+R`]('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!4E]6
M15)"051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2!["G)E
M<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ4&]D
M4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*)&9H+3YP<FEN="@D;&]A
M9&5R*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!L;V%D97(@=&\@(B1O=70B
M.B`D(5T["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(96%V>3L*4$%2
M.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN8R@I.PIR97%U
M:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)31%!!3B\@?0IG
M<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I9SHZ
M0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF:6=[
M<')I=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R96%C
M:"!`:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,;V%D
M97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI8E]E
M>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@("`@("`@(R!86%@@
M;&EB7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF;W)E86-H(&UY("1K
M97D@*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D;F%M92D["FEF
M("AD969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&ME
M>7TI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F8RD["GT*96QS92![
M"F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<421D
M:7)<12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("0Q*3L*
M;&%S=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD?"D@
M>PII9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI('L**"1F:6QE+"`D
M;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D,2(I
M('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0["GT*
M?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D=W)I
M='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I;&4@
M/7X@+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C:VEN
M9R!&24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65](#H@
M)&9I;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I*2!["B1C;VYT
M96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N9&5F
M.PIO<&5N(&UY("1T:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A;B=T
M(')E860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD=&@^.PIC;&]S92`D
M=&@["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1C;VYT96YT+"`B
M/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]615)"051)37T@86YD
M("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z.E!A
M=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D;F%M
M92D["GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H)&YA
M;64I("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z.F-O
M;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C:R@G3B<L(&QE;F=T
M:"@D8V]N=&5N="DI+`HD8V]N=&5N="D*;W(@9&EE('%Q6T5R<F]R('=R:71I
M;F<@96UB961D960@1DE,12!T;R`B)&]U="(Z("0A73L*;W5T<RAQ<5M7<FET
M=&5N(&%S("(D;F%M92)=*3L*?0I]"FEF("@D>FEP*2!["B1Z:7`M/G=R:71E
M5&]&:6QE2&%N9&QE*"1F:"D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2PIO<B!D
M:64@<7%;17)R;W(@=W)I=&EN9R!Z:7`@<&%R="!O9B`B)&]U=")=.PI]"FEF
M("@D;65T85]P87)[8VQE86Y]*2!["B1F:"T^<')I;G0H(EPP0TQ%04XB*3L*
M?0IE;'-E('L*:68@*"%D969I;F5D("1C86-H95]N86UE*2!["FUY("1C='@@
M/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIS965K("1F:"P@,"P@,#L*)&-T>"T^
M861D9FEL92@D9F@I.PIS965K("1F:"P@,"P@,CL*)&-A8VAE7VYA;64@/2`D
M8W1X+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A8VAE
M7VYA;64L(#`L("1C86-H95]N86UE7W-I>F4["B1C86-H95]N86UE("X](")<
M,"(@>"`H)&-A8VAE7VYA;65?<VEZ92`M(&QE;F=T:"`D8V%C:&5?;F%M92D[
M"B1F:"T^<')I;G0H)&-A8VAE7VYA;64L"B)<,$-!0TA%(BD["GT*;7D@)&]F
M9G-E="`]("1F:"T^=&5L;"`M(&QE;F=T:"@D;&]A9&5R*3L*)&9H+3YP<FEN
M="AP86-K*"=.)RP@)&]F9G-E="DL"B1005)?34%'24,I.PHD9F@M/F-L;W-E
M(&]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('1R86EL97(@;V8@(B1O=70B.B`D
M(5T["F-H;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL87-T('5N;&5S<R!D
M969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q=6ER92!0
M05(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR97%U:7)E
M($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY("1F:"`]
M($E/.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@("`@("`@("`@("`@("`C
M($%R8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ2&%N9&QE"B1F:"T^
M9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q<5MF9&]P96XH*2!F
M86EL960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H+7,@7T9(
M*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH)'II<"T^<F5A9$9R
M;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E.CI::7`Z
M.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E
M("(D<')O9VYA;64B73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H-C0@
M*B`Q,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@)'II<#L*)%!!4CHZ3&EB
M0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["F]U=',H(D5X=')A8W1I;F<@>FEP
M+BXN(BD["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]*2![(",@<VAO=6QD
M(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY("1M96UB97(@*"`D
M>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB97(M/FES1&ER96-T;W)Y
M.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M93L*;F5X="!U
M;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI8B\**#\Z)$-O;F9I
M9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E>'1R
M86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M/F-A
M=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H+68@
M)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E9%-I
M>F4H*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I;F-E
M(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L<V4@
M>PIO=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D97-T
M7VYA;64B+BXN72D["B@D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E
M<W1?;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@<7%;
M17)R;W(@97AT<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N86UE(ET[
M"F-H;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"(["GT*
M?0I]"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2!["F1I
M92`\/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP87(@
M72!;("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O9W)A
M;2YP;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*)$5.
M5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`05)'
M5BD["GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E<75I
M<F4@1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D97AT
M*2`]($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN*B<I
M*3L*<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H)'!A
M=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E<G)O
M<@I]"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q=6ER
M92!$>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R:6-T
M.PIR97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@0V%R
M<#L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q=6ER
M92!%>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I<F4@
M1F-N=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3<&5C
M.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER92!)
M3SHZ2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R97-S
M.CI:;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z
M.E-(03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER
M92!005(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PIR
M97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@871T
M<FEB=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U:7)E
M(%=I;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE=F%L
M('L@<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![(')E
M<75I<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E<75I
M<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@>R!R
M97%U:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD969I
M;F5D("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK
M*2\I('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY("1P
M871H("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%
M35!$25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI('L*
M;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD("UW
M("1P871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P=W5I
M9"D@/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E9FEN
M960H)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN,S(Z
M.DQO9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S97)N
M86UE(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U5315).
M04U%?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]"FUY("1S=&UP9&ER
M(#T@(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U<V5R
M;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER.PII
M9B`H)$5.5GM005)?0TQ%04Y]*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"([
M"GT*96QS92!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@;W(@
M9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*:68@*"AM>2`D
M;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS965K
M("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N9W1H
M*")<,$-!0TA%(BDL(#`["FUY("1B=68["G)E860@)&9H+"`D8G5F+"!L96YG
M=&@H(EPP0T%#2$4B*3L*:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E96L@
M)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG=&@H
M(EPP0T%#2$4B*0HM("1C86-H95]N86UE7W-I>F4L(#`["G)E860@)&9H+"`D
M8G5F+"`D8V%C:&5?;F%M95]S:7IE.PHD8G5F(#U^(',O7#`O+V<["B1C86-H
M95]D:7(@/2`B8V%C:&4M)&)U9B(["GT*?0IC;&]S92`D9F@["GT*:68@*"$D
M8V%C:&5?9&ER*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["B1%3E9[4$%2
M7T-,14%.?2`](#$["GT*)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?21C86-H
M95]D:7(B.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*)$5.5GM005)?5$5-4'T@
M/2`D<W1M<&1I<CL*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6>U!!
M4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U8B!?
M<V%V95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`7SL*
M;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H+64@
M)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B.PIM
M>2`D9F@["BAO<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE*0HF)B`H<')I
M;G0@)&9H("1C;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*;W(@9&EE('%Q6T5R
M<F]R('=R:71I;F<@(B1T96UP;F%M92(Z("0A73L*8VAM;V0@)&UO9&4L("1T
M96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE+"`D
M9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D9G5L
M;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D14Y6
M>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O*"XK
M*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*:68@
M*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[4$%2
M7U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA;64L
M(')I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II9B`H
M(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1S>7-[
M7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R;V=N
M86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA;64D
M<WES>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T=7)N
M.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A=&A?<V5P
M?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!4E]4
M14U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S+UQ1
M)'-Y<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D
M96QI;7TD<')O9VYA;64D<WES>U]E>&5](BDL(&QA<W0I"FEF("US("(D9&ER
M)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N86UE
M(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@+7,@
M(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O
M9VYA;64@>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345]
M.PII9B`H:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#P@,"D@>PHD
M<')O9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W
M9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@
M*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z
M(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN
M/UQ1)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6>U!!
M4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E;G8@
M>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E='5R
M;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF;W(@
M*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!344@
M*2`I('L*9&5L971E("1%3E9[(E!!4E\D7R)].PI]"F9O<B`H<7<O(%1-4$1)
M4B!414U0($-,14%.($1%0E5'("\I('L*)$5.5GLB4$%27R1?(GT@/2`D14Y6
M>R)005)?1TQ/0D%,7R1?(GT@:68@97AI<W1S("1%3E9[(E!!4E]'3$]"04Q?
M)%\B?3L*?0II9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[4$%2
M7T-,14%.?3L*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*:68@
M*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP<FEN
M="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I9SL*
M<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF:6=[
M)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I
M=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP86-K
M86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2.CIF
M:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A<BYP
M;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C
M:"!F:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE.PID
M;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+7T58
M251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/4B`]
M("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2(#U^
M+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@)#HZ
M7U]%4E)/4CL*,3L*7U]%3D1?7PH``$-!0TA%``!P97)L>'-I+F,````!&P,[
M$`$``"$```#D?___0`$```2`__\(#```1(+__RP!``!T@___9`$``%2$___<
M`0``Y(3__R@"``!DAO__W`(``)2'__]D`P``Y(G__^0#``"TBO__<`0``!2+
M___`!```5(O__^0$```4C/__=`4``"2,__^(!0``5([__X0&``"4CO__M`8`
M`!20__\T!P``%)3__W0'``"$E/__D`<``+24__^\!P``M)7__T`(``#4G/__
MT`H``.2>__^@"P``M9___U`,``"YG___9`P``+V?__]X#```P9___XP,``#4
MG___H`P``+2K___D#```!*S__P`-``#4K?__4`T``.2O__^0#0``%+#__[0-
M```4``````````%Z4@`!?`@!&PP$!(@!```0````'````!"!__\P`````$0'
M""`````P````G'[__R``````#@A&#@Q*#PMT!'@`/QH[*C(D(G0```!4````
M"(+__]P`````00X(A0),#@R'`T$.$(8$00X4@P5##C`"20XT00XX1`X\00Y`
M20XP60H.%$/##A!!Q@X,0<<."$'%#@1'"W0*#A1#PPX00<8.#$''#@A!Q0X$
M1@M##A1%PPX00<8.#$''#@A!Q0X$`$@```#,````<(+__X(`````00X(A@),
M#@R#`T4.B`%1#HP!00Z0`4D.@`%A#H@!0@Z,`4$.D`%)#H`!8`H.#$'##@A!
MQ@X$00L```"P````&`$``+2"__]^`0```$$."(4"00X,AP-!#A"&!$P.%(,%
M2`Z@`6P*#A1!PPX00<8.#$''#@A!Q0X$20M##JP!00ZP`4L.H`%'#JP!00ZP
M`4D.H`%Q#JP!00ZP`6D.K`%'#J@!10ZL`4$.L`%!#K0!0@ZX`4(.O`%!#L`!
M20ZH`44.K`%!#K`!20Z@`5@.K`%!#K`!20Z@`4,.K`%!#K`!20Z@`4X.K`%!
M#K`!4@Z@`0"$````S`$``("#__\M`0```$$."(4"00X,AP-##A"&!$$.%(,%
M40[`@`15#L2`!$(.R(`$0@[,@`1!#M"`!'L.P(`$5@[$@`1!#LB`!$$.S(`$
M1`[0@`1)#L2`!$4.R(`$00[,@`1$#M"`!$D.P(`$`E(*#A1!PPX00<8.#$''
M#@A!Q0X$1@L`?````%0"```HA/__2`(```!!#@B%`DP.#(<#0PX0A@1!#A2#
M!44.3%<.4$D.0%H.3$,.4$T.0`)4#DQ+#E!+#D`":0H.%$'##A!!Q@X,0<<.
M"$'%#@1)"P)D#DQ!#E!5#D!2#D1%#DA%#DQ!#E!5#D!7#DA!#DQ!#E!/#D``
M``"(````U`(``/B%___(`````$$."(4"00X,AP-,#A"&!$,.%(,%1@[$(%@.
MR"!+#LP@00[0($D.P"!=#L0@1`[(($$.S"!##M`@20[`(%8*#A1!PPX00<8.
M#$''#@A!Q0X$1`M+#L0@00[(($(.S"!!#M`@20[`($L.Q"!!#L@@00[,($,.
MT"!)#L`@`$P```!@`P``/(;__UP`````00X(A@)!#@R#`TX.$$T.%$P.&$$.
M'$,.($<.'$<.&$(.'$$.($$.)$<.*$<.+$,.,$D.#$'##@A!Q@X$````(```
M`+`#``!,AO__-`````!-#A!'#A1,#AA!#AQ)#B!)#@0`C````-0#``!HAO__
MO`````!!#@B%`D,.#(<#00X0A@1!#A2#!5$.L$!8#KA`1PZ\0$$.P$!'#L1`
M10[(0$(.S$!%#M!`20ZT0$4.N$!(#KQ`00[`0$D.L$!'#KQ`1@[`0$L.L$!'
M#K1`00ZX0$$.O$!!#L!`3@ZP0%8*#A1#PPX00<8.#$''#@A!Q0X$00L`$```
M`&0$``"8AO__!0````````#X````>`0``)2&__\G`@```$$."(4"00X,AP-!
M#A"&!$$.%(,%40ZP(74.N"%"#KPA0P[`(4D.L"%M"@X40<,.$$'&#@Q!QPX(
M0<4.!$<+0PZ\(4$.P"%'#KPA00ZX(4L.O"%!#L`A5PZP(6`.N"%!#KPA00[`
M(4D.L"%+#KPA00[`(4\.L"%=#KPA00[`(4L.L"%'#KPA00[`(5$.L"%.#K0A
M00ZX(40.O"%!#L`A1`[$(44.R"%"#LPA2`[0(4D.N"%$#KPA00[`(4D.L"%7
M#K@A1`Z\(4(.P"%+#K`A`D8.N"%"#KPA00[`(4D.L"%4#KPA00[`(4D.L"$L
M````=`4``,B'__\V`````$$."(8"3`X,@P-##AA&#AQ!#B!7#@Q!PPX(0<8.
M!`!\````I`4``-B'__]T`0```$$."(4"3`X,AP-!#A"&!$$.%(,%1@ZP(&H.
MO"!!#L`@3`ZP(`)&#K@@0@Z\($$.P"!)#K`@3PZ\($$.P"!)#K`@6@H.%$'#
M#A!!Q@X,0<<."$'%#@1+"V,*#A1#PPX00<8.#$''#@A!Q0X$1PL``#P````D
M!@``V(C___P#````00X(A0)!#@R'`T$.$(8$3`X4@P5##D`#N`,*#A1'PPX0
M0<8.#$''#@A!Q0X$10L8````9`8``)B,__]A`````$T.(`)&"@X$10L`*```
M`(`&``#LC/__+0````!!#@B#`DX.$$D.'$$.($D.$$D."$'##@0```"`````
MK`8``/",___Y`````$$."(4"00X,AP-!#A"&!$P.%(,%0PY``D(*#A1#PPX0
M0<8.#$''#@A!Q0X$1@M##DA!#DQ!#E!)#D!4"@X40<,.$$'&#@Q!QPX(0<4.
M!$H+0PY,00Y09PY$20Y(00Y,1PY000Y40@Y80@Y<00Y@30Y```",`@``,`<`
M`&R-__\>!P```$$."(4"00X,AP-!#A"&!$$.%(,%40Z0`@+0#IP"00Z@`DD.
MD`)5#IP"00Z@`DT.D`)##IP"00Z@`D\.D`)>#IP"1PZ@`D$.I`)"#J@"0@ZL
M`D$.L`)0#I`"`D@.G`)!#J`"2PZ0`G\.G`)!#J`"1PZ<`D0.H`)H#J0"00ZH
M`DL.K`)'#K`"0PZT`D(.N`)"#KP"00[``DD.F`)%#IP"00Z@`DD.D`)9#I@"
M00Z<`D0.H`)+#I`"1PZ<`D0.H`)&#J0"1PZH`DH.K`)"#K`"20Z0`EH*#A1#
MPPX00<8.#$''#@A!Q0X$2`M##IP"1`Z@`DL.D`("60Z<`D$.H`)8#I`"2PZ8
M`D$.G`)&#J`"3`Z0`DL.G`)$#J`"2PZ0`D@.F`)'#IP"30Z@`DD.D`(":PZ4
M`D$.F`)*#IP"1`Z@`D$.I`)"#J@"0@ZL`D$.L`)/#I`"7PZ<`D$.H`)!#J0"
M00ZH`D0.K`)$#K`"1`ZT`D(.N`)"#KP"00[``DD.D`)##I@"10Z<`D$.H`))
M#I`"2`Z<`D0.H`)<#I`"`D<.G`)(#J`"2PZ0`EL.G`)!#J`"2PZ0`DP*#I@"
M00Z<`D0.H`)&#J0"4@M##IP"00Z@`D@.D`)A"@X40\,.$$'&#@Q!QPX(0<4.
M!$T+6PZ8`D(.G`)$#J`"20Z0`F8.E`)$#I@"00Z<`D4.H`)-#I0"0@Z8`D@.
MG`)%#J`"30Z0`F<.E`)/#I@"00Z<`D$.H`)-#I0"0@Z8`D@.G`)%#J`"50ZD
M`D<.J`)$#JP"1`ZP`D$.M`)"#K@"0@Z\`D$.P`))#I`"8`Z<`D$.H`)-#I`"
M3`Z8`D<.G`))#J`"30Z0`@```,P```#`"0``_)'__P$"````00X(A0)!#@R'
M`T$.$(8$00X4@P5.#C`"2@XT0@XX00X\00Y`2PXP1PXT1PXX10X\0PY`3@XP
M4@XT0@XX00X\0PY`2PXP7`XT00XX10X\0PY`20XP0PXT2`XX00X\0PY`3PXT
M00XX00X\0PY`70XP3PXX10X\00Y`3`XP8@H.%$'##A!!Q@X,0<<."$'%#@1)
M"T,*#CQ&#D!(#C!/#CQ'#D!)"W,..$$./$,.0$D.,$8*#C1'#CA"#CQ'#D!&
M"P!D````D`H``#R3___1`````$$."(<"00X,A@-,#A"#!$,.+%$.,$8.+$$.
M*$(.+$,.,&L.('<*#A!!PPX,0<8."$''#@1%"T,.)$@.*$$.+$$.,$D.(%0*
M#A!#PPX,0<8."$''#@1%"T0```#X"@``]'/__SP"````1`P!`$D0!0)U`$$0
M!P)U?$00!@)U>!`#`G5T3`\#=7`&`V\!"L$,`0!!PT'&0<=!Q4,,!`1!"Q``
M``!`"P``79/__P0`````````$````%0+``!-D___!``````````0````:`L`
M`#V3__\$`````````!````!\"P``+9/__P0`````````0````)`+```LD___
MWPL```!!#@B%`D$.#(<#00X0A@1!#A2#!4@.D`,#R0L*#A1!PPX00<8.#$''
M#@A!Q0X$00L````8````U`L``,B>__]0`````$T.'$(.(`)`#@0`3````/`+
M``#\GO__QP$```!!#@B%`D$.#(<#00X0A@1,#A2#!4,.,`,L`0H.%$'##A!!
MQ@X,0<<."$'%#@1&"P)##C1##CA%#CQ!#D!2#C`\````0`P``'R@__\0`@``
M`$L."(4"00X,AP-!#A"&!$$.%(,%0PXP`RL!"@X40<,.$$'&#@Q!QPX(0<4.
M!%`+(````(`,``!,HO__+0````!1#A!&#A1$#AA$#AQ$#B!)#@0`&````*0,
M``!8HO__&0````!%#@A!#@1-#A``````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
ML",``&`C```!````\P(```$```#^`@``'0```(H$```,`````"````T```!\
M4```&0```-B]```;````!````!H```#<O0``'`````0```#U_O]O-`(```4`
M``!H"0``!@```(@#```*````K00```L````0````%0`````````#````Z+X`
M``(````(````%````!$````7````I!$``!$```!T#P``$@```#`"```3````
M"````!X````(````^___;P$```C^__]OU`X``/___V\!````\/__;Q8.``#Z
M__]O!0``````````````````````````````````````````````````````
M``#@O0````````````!&(```````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````!0(`````````3````*4$%2+G!M"@````#(8@``1T-#.B`H1TY5
M*2`Q,RXQ+C$@,C`R,S`T,CD``"YS:'-T<G1A8@`N:6YT97)P`"YN;W1E+F=N
M=2YB=6EL9"UI9``N;F]T92YG;G4N<')O<&5R='D`+FYO=&4N04))+71A9P`N
M9VYU+FAA<V@`+F1Y;G-Y;0`N9'EN<W1R`"YG;G4N=F5R<VEO;@`N9VYU+G9E
M<G-I;VY?<@`N<F5L+F1Y;@`N<F5L+G!L=``N:6YI=``N=&5X=``N9FEN:0`N
M<F]D871A`"YE:%]F<F%M95]H9'(`+F5H7V9R86UE`"YI;FET7V%R<F%Y`"YF
M:6YI7V%R<F%Y`"YD>6YA;6EC`"YG;W0`+F1A=&$`+F)S<P`N8V]M;65N=```
M```````````````````````````````````````````````````````+````
M`0````(```"T`0``M`$``!,```````````````$`````````$P````<````"
M````R`$``,@!```D```````````````$`````````"8````'`````@```.P!
M``#L`0``*```````````````!``````````Y````!P````(````4`@``%`(`
M`"````````````````0`````````1P```/;__V\"````-`(``#0"``!4`0``
M!@`````````$````!````%$````+`````@```(@#``"(`P``X`4```<````!
M````!````!````!9`````P````(```!H"0``:`D``*T$``````````````$`
M````````80```/___V\"````%@X``!8.``"\````!@`````````"`````@``
M`&X```#^__]O`@```-0.``#4#@``H`````<````!````!`````````!]````
M"0````(```!T#P``=`\``#`"```&``````````0````(````A@````D```!"
M````I!$``*01```(````!@```!8````$````"````(\````!````!@`````@
M````(```)```````````````!`````````"*`````0````8````P(```,"``
M`"```````````````!`````$````E0````$````&````4"```%`@```I,```
M```````````0`````````)L````!````!@```'Q0``!\4```&```````````
M````!`````````"A`````0````(`````8````&```$I```````````````0`
M````````J0````$````"````3*```$R@```4`0`````````````$````````
M`+<````!`````@```&"A``!@H0``P`P`````````````!`````````#!````
M#@````,```#8O0``V+T```0```````````````0````$````S0````\````#
M````W+T``-R]```$```````````````$````!````-D````&`````P```."]
M``#@O0``"`$```<`````````!`````@```#B`````0````,```#HO@``Z+X`
M`!@!``````````````0````$````YP````$````#`````,````#````8````
M```````````$`````````.T````(`````P```!C````8P```%```````````
M````!`````````#R`````0```#``````````&,```!L```````````````$`
M```!`````0````,``````````````#/```#[```````````````!````````
!````
