package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.42.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.42.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````%!D``#0```#4H0````0`!30`(``*
M`"@`'``;``$``'"LC@``K(X``*R.``!P````<`````0````$````!@```#0`
M```T````-````$`!``!``0``!`````0````#````F`$``)@!``"8`0``&0``
M`!D````$`````0````$```````````````````!`CP``0(\```4`````$```
M`0```*R=``"LG0``K)T``)P"``"P`@``!@`````0```"````M)T``+2=``"T
MG0``&`$``!@!```&````!`````0```!T`0``=`$``'0!```D````)`````0`
M```$````!````""/```@CP``((\``"`````@````!`````0```!1Y71D````
M```````````````````````&````$````%+E=&2LG0``K)T``*R=``!4`@``
M5`(```0````!````!````!0````#````1TY5`/TQ9J)G-AKRK$7F[V5#:NC)
M:B?*+VQI8B]L9"UL:6YU>"UA<FUH9BYS;RXS`````#L```!-````"`````@`
M```!D!`$)4`$@"B```6@AE"$@($``@LH!(PX@!)!B@``"`````!-````3P``
M`%````````````````````!2`````````%,```!4````50``````````````
M5@```%<`````````````````````````6````%D```!:`````````%L`````
M``````````````!<````70```%X```````````````````!?````8```````
M``!B`````````&,`````````9```````````````````````````````````
M`&4```````````````````!F`````````&<```````````````````!I````
M?JF<FUGOEWEK"4/6=E0\E"'_Y`X-.I<+P^4_+(D'<-/9<5@<K4OCP&L:GE#5
M9-1\F4_#FR_%F[8S8MOM:W^:?`&O-.B[XY)\LP&+;T+ZUX=#1=7L9Z_,9A6,
M)<RCX/M)S]Y'+>-'Q@'@IP(*;\7?'[[:5QO[H6Q@````````````````````
M```````H$P````````,`"@```````*`````````#`!<`"0(`````````````
M$@```'$``````````````!(```#A`@`````````````2````K`(`````````
M````$@```)X"`````````````!(```![```````````````2````B@$`````
M````````$@```!T#`````````````!(```"'```````````````2`````0``
M````````````(````)<"`````````````!(```"Y```````````````2````
M80``````````````$@```*4"`````````````!(```#G```````````````2
M````U`$`````````````$@```!```````````````"````#<````````````
M```2````M`(`````````````$@```'8``````````````!(```#W`0``````
M```````2````NP(`````````````$@```"P``````````````"````!A`@``
M```````````2````_`$`````````````$@```%,``````````````!(````G
M`0`````````````6````!`$`````````````$@```&H!`````````````!(`
M``"H```````````````2````]0``````````````$@```.H!````````````
M`!(```#<`0`````````````2````P@(`````````````$@````($````````
M`````!(```#-```````````````2````\0(`````````````$0```,T!````
M`````````!(````+`P`````````````1````#P(`````````````$@```)H`
M`````````````!(```#$`0`````````````2````O0$`````````````$@``
M`%X!`````````````!(```"K`0`````````````2````(`$`````````````
M$@```)4!`````````````!(```!2`@`````````````2````XP$`````````
M````$@```/$!`````````````!(```!@`@`````````````2````^0(`````
M````````$@```#H!`````````````!(````>`@`````````````2````%`$`
M````````````$@```"4"`````````````!(```#O`@`````````````1````
M>@(`````````````$0```$8``````````````!$```"S`0`````````````2
M````3P$`````````````$0```#8"`````````````!(```#G`@``````````
M```2````P@``````````````$@```$0"`````````````!(````N`@``````
M```````2````>P$`````````````$@```&@"`````````````!(````T`P``
M```````````2````@0(`````````````$@```,D"`````````````!(```"F
M`0`````````````2````T`(`````````````$@```(@"`````````````"(`
M``#P`P``,#```"0````2``P`5`0``%R@````````$``8`"8$````H```````
M`!``%P`S!```2*`````````0`!@`J`0``#@C``"8`@``$@`,`"@$````H```
M`````"``%P!,!```R"$``%0````2``P`M00``$2@```$````$0`7`&@$``!(
MH````````!``&``7!```D$P```0````1``X`Y@,```Q+``!H`0``$@`,`*`#
M```$.0``(`,``!(`#`"_!```F"T``'0````2``P`@@,``#0C```$````$@`,
M`"P$```4&0```````!(`#`!U`@``&!8``/P"```2``P`700``%R@````````
M$``8`'T$``!<H````````!``&`"V`P``^"P``*`````2``P`000``.A)```D
M`0``$@`,`(\$``!(H````````!``%P#:`P```"8``+0!```2``P`=`0``)1)
M``!4````$@`,`,<$``"T)P``1`4``!(`#`""!```C#$``'@'```2``P`DP,`
M`-`E```P````$@`,`,0#```<(@``&`$``!(`#``+!```)#P``#P!```2``P`
M7`0``%R@````````$``8`)8$``!4,```.`$``!(`#```7U]G;6]N7W-T87)T
M7U\`7TE435]D97)E9VES=&5R5$U#;&]N951A8FQE`%])5$U?<F5G:7-T97)4
M34-L;VYE5&%B;&4`4$Q?9&]?=6YD=6UP`%!E<FQ?<WES7W1E<FT`4&5R;%]G
M=E]F971C:'!V`'!E<FQ?9G)E90!097)L7V=E=%]S=@!097)L7V%T9F]R:U]U
M;FQO8VL`<&5R;%]D97-T<G5C=`!097)L7VYE=UA37V9L86=S`'!E<FQ?<G5N
M`%!E<FQ?;F5W6%,`4&5R;%]S=E]S971P=FX`<&5R;%]A;&QO8P!097)L7W-V
M7W-E='!V`'!E<FQ?8V]N<W1R=6-T`&)O;W1?1'EN84QO861E<@!097)L7W-A
M9F5S>7-M86QL;V,`4$Q?8W5R<F5N=%]C;VYT97AT`%!E<FQ?8W)O86M?;F]C
M;VYT97AT`%!,7VUE;6]R>5]W<F%P`%!E<FQ?;6=?<V5T`%!E<FQ?871F;W)K
M7VQO8VL`4&5R;%]S>7-?:6YI=#,`<&5R;%]P87)S90!?7W-T86-K7V-H:U]F
M86EL`&5X:70`;'-E96LV-`!R96%D9&ER-C0`<W1R9'5P`&-L;W-E9&ER`&]P
M96XV-`!S=')R8VAR`'5N;&EN:P!S=')T;VL`<W1R;&5N`')M9&ER`')E860`
M7U]M96UC<'E?8VAK`&UK9&ER`%]?=F9P<FEN=&9?8VAK`&=E='!I9`!R96%D
M;&EN:P!R96%L;&]C`%]?<W!R:6YT9E]C:&L`7U]M96UM;W9E7V-H:P!?7V9P
M<FEN=&9?8VAK`&QS=&%T-C0`7U]L:6)C7W-T87)T7VUA:6X`<W1D97)R`&UE
M;6-H<@!?7V-X85]F:6YA;&EZ90!S=')C:'(`;65M8VUP`&UE;7-E=`!O<&5N
M9&ER`&=E='5I9`!A8V-E<W,`;65M8W!Y`'-T<F-M<`!?7V5R<FYO7VQO8V%T
M:6]N`&%B;W)T`'-T<FYC;7``7U]E;G9I<F]N`%]?<F5G:7-T97)?871F;W)K
M`%]?<W1A8VM?8VAK7V=U87)D`%]?865A8FE?=6YW:6YD7V-P<%]P<C``7U]A
M96%B:5]U;G=I;F1?8W!P7W!R,0!L:6)P97)L+G-O`&QI8F,N<V\N-@!L9"UL
M:6YU>"UA<FUH9BYS;RXS`&QI8F=C8U]S+G-O+C$`<&%R7V-U<G)E;G1?97AE
M8P!P87)?8F%S96YA;64`6%-?26YT97)N86QS7U!!4E]"3T]4`'!A<E]E;G9?
M8VQE86X`<&%R7V-U<G)E;G1?97AE8U]P<F]C`'!A<E]D:7)N86UE`'-H85]F
M:6YA;`!G971?=7-E<FYA;65?9G)O;5]G971P=W5I9`!P87)?8VQE86YU<`!?
M24]?<W1D:6Y?=7-E9`!?7V1A=&%?<W1A<G0`7U]B<W-?<W1A<G1?7P!S:&%?
M=7!D871E`'AS7VEN:70`7U]E;F1?7P!?7V)S<U]E;F1?7P!?7V)S<U]S=&%R
M=`!S:&%?:6YI=`!?96YD`'!A<E]M:W1M<&1I<@!?961A=&$`<&%R7W-E='5P
M7VQI8G!A=&@`<&%R7V9I;F1P<F]G`&UY7W!A<E]P;`!P87)?9&EE`'!A<E]I
M;FET7V5N=@!'3$E"0U\R+C0`1T-#7S,N-0!'3$E"0U\R+C,T`$=,24)#7S(N
M,S,`+W5S<B]L:6(O<&5R;#4O-2XT,B]C;W)E7W!E<FPO0T]210``````````
M`@`!``(``@`"``$``0`#``$``0`"``$``0`"``$``@`!``$``@`"``(``@`!
M``0``@`!``$``0`!``$``0`"``(``@`"``$``@`"``4``@`!``(``@`!``(`
M`@`"``(``@`"``0``@`!``(``0`"``(``@`!``(``0`"``(``0`"``(``0`&
M``,``@`"``(``@`"``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$`
M`0`!``$``0`!``$``0`!``$``0`!``$``0`!`````0`!`&`#```0````(```
M`!1I:0T```4`U`0````````!``$`=`,``!`````@````529Y"P```P#>!```
M``````$``P!6`P``$`````````"TD98&```&`.8$```0````LY&6!@``!`#Q
M!```$````!1I:0T```(`U`0```````"LG0``%P```+"=```7````_)\``!<`
M```$H```%P```!2@```7````&*```!<````@H```%P```"2@```7````**``
M`!<````LH```%P```#"@```7````.*```!<````\H```%P```$2@```7````
MR)\``!4+``#,GP``%0P``-"?```5$P``U)\``!49``#8GP``$QT``-R?```5
M'@``X)\``!4?``#DGP``%2<``.B?```5*0``[)\``!4\``#PGP``%3T``/2?
M```5/P``^)\``!5,``#8G@``%@,``-R>```6!```X)X``!8%``#DG@``%@8`
M`.B>```6!P``[)X``!8(``#PG@``%@D``/2>```6#```^)X``!8-``#\G@``
M%@X```"?```6#P``!)\``!80```(GP``%A$```R?```6$@``$)\``!84```4
MGP``%A4``!B?```6%@``')\``!87```@GP``%A@``"2?```6&@``*)\``!8;
M```LGP``%AP``#"?```6(```-)\``!8A```XGP``%B(``#R?```6(P``0)\`
M`!8D``!$GP``%B4``$B?```6)@``3)\``!8H``!0GP``%BH``%2?```6*P``
M6)\``!8L``!<GP``%BT``&"?```6+@``9)\``!8O``!HGP``%C```&R?```6
M,0``<)\``!8R``!TGP``%C,``'B?```6-```?)\``!8U``"`GP``%C8``(2?
M```6-P``B)\``!8X``",GP``%CD``)"?```6.@``E)\``!8^``"8GP``%D``
M`)R?```600``H)\``!9"``"DGP``%D,``*B?```61```K)\``!9%``"PGP``
M%D8``+2?```62```N)\``!9)``"\GP``%DH``,"?```62P``Q)\``!9,```(
M0"WIB0$`ZPB`O>@$X"WE!."?Y0[@C^`(\+[EB(L```#&C^((RHSBB/N\Y0#&
MC^((RHSB@/N\Y0#&C^((RHSB>/N\Y0#&C^((RHSB</N\Y0#&C^((RHSB:/N\
MY0#&C^((RHSB8/N\Y0#&C^((RHSB6/N\Y0#&C^((RHSB4/N\Y0#&C^((RHSB
M2/N\Y0#&C^((RHSB0/N\Y0#&C^((RHSB./N\Y0#&C^((RHSB,/N\Y0#&C^((
MRHSB*/N\Y0#&C^((RHSB(/N\Y0#&C^((RHSB&/N\Y0#&C^((RHSB$/N\Y0#&
MC^((RHSB"/N\Y0#&C^((RHSB`/N\Y0#&C^((RHSB^/J\Y0#&C^((RHSB\/J\
MY0#&C^((RHSBZ/J\Y0#&C^((RHSBX/J\Y0#&C^((RHSBV/J\Y0#&C^((RHSB
MT/J\Y0#&C^((RHSBR/J\Y0#&C^((RHSBP/J\Y0#&C^((RHSBN/J\Y0#&C^((
MRHSBL/J\Y0#&C^((RHSBJ/J\Y0#&C^((RHSBH/J\Y0#&C^((RHSBF/J\Y0#&
MC^((RHSBD/J\Y0#&C^((RHSBB/J\Y0#&C^((RHSB@/J\Y0#&C^((RHSB>/J\
MY0#&C^((RHSB</J\Y0#&C^((RHSB:/J\Y0#&C^((RHSB8/J\Y0#&C^((RHSB
M6/J\Y0#&C^((RHSB4/J\Y0#&C^((RHSB2/J\Y0#&C^((RHSB0/J\Y0#&C^((
MRHSB./J\Y0#&C^((RHSB,/J\Y0#&C^((RHSB*/J\Y0#&C^((RHSB(/J\Y0#&
MC^((RHSB&/J\Y0#&C^((RHSB$/J\Y0#&C^((RHSB"/J\Y0#&C^((RHSB`/J\
MY0#&C^((RHSB^/F\Y0#&C^((RHSB\/F\Y0#&C^((RHSBZ/F\Y0#&C^((RHSB
MX/F\Y0#&C^((RHSBV/F\Y0#&C^((RHSBT/F\Y0#&C^((RHSBR/F\Y0#&C^((
MRHSBP/F\Y0#&C^((RHSBN/F\Y0#&C^((RHSBL/F\Y7!`+>D!RJ#C#,!-X,@/
MC.6(PI_E*-!-XH0RG^4,P(_@@$*?Y1P`C>4<`(WB&!"-Y1@0C>(4((WE%""-
MX@1`C^`#,)SG`#"3Y20PC>4`,*#CU?__ZU`BG^50,I_E`B"4YPP@C>4"$*#A
M`S"4YQ`PC>4#`*#A>`T`ZS0RG^4#,)3G`%#3Y0``5>-2```*<"\=[B`RG^4<
M`)WE`U"`X@0`@.(#,)_G!P%PXP,@DN>F,]+E`S"#XZ8SPN5C``"*``&@X:'_
M_^OP$9_E\"&?Y0`PH.$!$(_@&$"=Y0(@C^``$)'E!`""Y0``E.4($(/ET!&?
MY1P@G>4!$(_@#!"#Y<01G^4!`%+C``"#Y0$0C^`$$(/E70``V@00A.(,P(/B
M`D&$X`3@D>0$X*SE`0!4X?O__QH#$(+BD$&?Y0``H.,!`8/G!2"@X01`C^``
M`(WE?!&?Y0P`E.4!$(_@!O__ZP``4.,I```*:%&?Y2!@C>(%4(_@#`"5Y4K_
M_^L`0*#A5`&?Y080H.$``(_@N```ZP``4.,@```*0#&?Y0,PC^`,`)/EY?[_
MZR#__^LP(9_E\#"?Y0(@C^`#,)+G`""3Y20PG>4"(#/@`#"@XS,``!H$`*#A
M*-"-XG"`O>C^_O_K`"&?Y0``4.,"((_@#`""Y2@```H3___K[#"?Y7`O'>X#
M,)_G`S"2YSY6P^6@___J#`"4Y>'^_^O2___J"`"5Y0``4.,!```*^`@`Z]G_
M_^JX`)_E!A"@X0``C^",``#K``!0XP@`A>72__\*]?__ZIP@G^5,,)_E`B"/
MX`,PDN<`()/E)#"=Y0(@,^``,*#C"@``&GPPG^5\`)_E``"/X`,PE.<,,(WE
M`Q"@X2G__^L$$*#CI___Z@$`H.-/___K$O__ZY"(```<`0``<(@``/P````4
M`0``)`$``!B)``!4B0``5(D``#0Y```<.0``Z(@``*@)``#`B```1#8``)2(
M``#\A@``2(@``+R'```P-0``6(8``"@!``#L-@```+"@XP#@H.,$$)WD#2"@
MX00@+>4$`"WE'*"?Y1@PC^(#H(K@`#"@XP0P+>4,`)_E``":YR#__^N#_O_K
M?(4``#`!```4,)_E%""?Y0,PC^`"()/G``!2XQ[_+P&)_O_J9(4````!```L
M`)_E+#"?Y0``C^`H()_E`S"/X```4^$"((_@'O\O`1@PG^4#,)+G``!3XQ[_
M+P$3_R_AO(8``+2&```PA0``!`$``#@`G^4X,)_E``"/X#0@G^4#,(_@`#!#
MX`(@C^"C'Z#A0Q&!X,$0L.$>_R\!&#"?Y0,PDN<``%/C'O\O`1/_+^%XA@``
M<(8``.R$```(`0``6#"?Y5@@G^4#,(_@`B"/X``PT^4``%/C'O\O$1!`+>D!
MRJ#C#,!-X/@/C.4T,)_E`S"2YP``4^,"```**#"?Y0,`G^?L_O_KQ___ZQPP
MG^4!(*#C`S"/X``@P^40@+WH*(8``*B$```L`0``K(4``-R%``#-___J\$<M
MZ0'*H.,,P$W@X`^,Y;PPG^6\()_E`S"/X`(PD^<`D)/E``!9XR<```H`4-#E
M`("@X0%PH.$``%7C/0!5$P%0H!,`4*`#$0``&@!`F>4``%3C#```"@E@H.$%
M(*#A"!"@X00`H.&G_O_K``!0XP(``!H%,-3G/0!3XPL```H$0+;E``!4X_/_
M_QH$`*#A\(>]Z`!0H.$!,/7E``!3XST`4Q/[__\:"%!%X.;__^H)8$;@`5"%
MX@5`A.!&8:#A!`"@X0!@A^7PA[WH"4"@X>W__^HDA```&`$``!!`+>D!RJ#C
M#,!-X(@/C.6,()_E<-!-XH@PG^4-$*#A`B"/X`!`H.$#,)+G`#"3Y6PPC>4`
M,*#C9/[_ZP``4.,``*`3!```&A`PG>4/.@/B"@I3XP$)4Q,*```*2""?Y4`P
MG^4"((_@`S"2YP`@D^5L,)WE`B`SX``PH.,'```:<-"-XA"`O>@"$*#C!`"@
MX0G^_^L0#V_AH`*@X>[__^H^_O_K0(,``!P!``#\@@``\$`MZ0'*H.,,P$W@
M<`^,Y6`AG^5\T$WB7#&?Y0!04.("((_@`S"2YP`PD^5T,(WE`#"@XP(```H`
M,-7E``!3XPH``!HT(9_E+#&?Y0(@C^`#,)+G`""3Y70PG>4"(#/@`#"@XT(`
M`!I\T(WB\("]Z+C]_^L`8%#B\?__"@8`H.$X_O_K``!0XRL```H3,-#E$T"`
MXBX`4^,"```:`3#4Y0``4^/T__\*$S#0Y2X`4^,F```*!0"@X>7]_^L`<*#A
M!`"@X>+]_^L`<(?@`G"'X@<`H.$"_O_KI#"?Y0<@H.$"$*#C`S"/X`1`C>4`
M4(WE`$"@X1[^_^L($(WB!`"@X<3]_^L!`'#C!`"@X0,```H8,)WE#SH#X@$)
M4^,2```*SOW_ZP0`H.&Q_?_K!@"@X0S^_^L``%#CT___&@8`H.';_?_K!0"@
MX?']_^N\___J`3#4Y2X`4^/5__\:`C#4Y0``4^/$__\*T?__ZJ/__^OK___J
MW?W_ZY""```<`0``9((``"0P``#P3RWI`<J@XPS`3>#0#XSE'"*?Y0S03>(8
M,I_E`4"@X0(@C^``4*#A#(*?Y0,PDN<(@(_@`#"3Y00PC>4`,*#C`##1Y3T`
M4^,!0($"!`"@X9W]_^L-$*#A`&"@X04`H.$A___K`)!0XD````J6_?_K``!6
MX3<``)K`,9_E`'"=Y0.`F.<'<:#A`##5Y0``4^,]`%,3`3"@$P`PH`,%```*
M!3"@X0$@\^4``%+C/0!2$_O__QH%,$/@`@"&X@!@F.4#`(#@I/W_ZP``4.,'
M`(;G%0``"@`PF.4%`*#A!Q"3YP$PT.0`,,'E/0!3XP``4Q,&```*`2"!X@$P
MT.0"$*#A`3#"Y```4^,]`%,3^?__&@$@H.$],*#C`##!Y0$PU.0!,.+E``!3
MX_O__QH0(9_E`#&?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@XS<``!H,T(WB
M\(^]Z`&02>(!,-3D`3#IY0``4^/[__\:[O__ZL@PG^4#@)CG`'"8Y0`PE^4`
M`%/C)@``"@<PH.$$(+/E"1"@X0&0B>(``%+C^O__&@,0@>(!L:#AF#"?Y0,P
MC^``()/E``!2XQ```!H+`*#A`2"@XP`@@^5C_?_K`*!0XM7__PH'$*#A"7&@
MX0LPH.$'(*#AB?W_ZP"@B.4`D(WE`9")X@`PH.,),8KGH?__Z@L0H.$'`*#A
M@_W_ZP"@H.$``%#C``"(Y</__PH)<:#A\?__Z@.0H.$(L*#CW?__ZDS]_^L(
M@0``'`$``/B````8`0``]'\``/R```#P02WI`<J@XPS`3>```(SE(``,Y0%P
MH.'\$)_E_#"?Y0':3>(!$(_@"-!-X@!@H.$!"HWB!`"`X@2`C>(#,)'G_R\`
MXP@0H.$`,)/E`#"`Y0`PH./(`)_E``"/X$G]_^L`0%#B!@``V@$P1.(#P(W@
M!"#<Y0``4N,7```:``!3XQ0``!J<,)_E<`\=[@<@H.$&$*#A`S"?YP,`D.?3
M_/_KA""?Y0$:C>)P,)_E!!"!X@(@C^`#,)+G`""3Y0`PD>4"(#/@`#"@XQ(`
M`!H!VHWB"-"-XO"!O>@#0*#A!""@X2\0H.,(`*#A0?W_ZP``4./C__\*-`"?
MY7#/'>X$,*#A"""@X080H.$``)_G``"<Y^;\_^OA___J__S_Z[A^```<`0``
MY"P``%1_```L?@``X'X``!!`+>D!RJ#C#,!-X.@/C.5P$)_E<$\=[FP@G^40
MT$WB:`"?Y0(@C^!D,)_E`1"?YP,PC^``()+G`0"4YU00G^4,((WE`1"/X`W]
M_^M($)_E2#"?Y0#`H.-$()_E`S"/X`$0G^<"((_@`0"4YS00G^4$P(WE`1"/
MX``PC>6L_/_K$-"-XA"`O>B(?@``A'T``!`!``"0;```U"L``%1^``#`+@``
M?!<``,`K```X`)_E"-!-XC0@G^5PSQWN,!"?Y0(@C^`L,)_E``"?YP,PC^`!
M()+G(!"?Y0``G.<!$(_@!""-Y0C0C>+E_/_J['T``.A\```0`0``]&L``#PK
M``#P0"WI`<J@XPS`3>```(SE`<I,X@``C.4P``SEX""?Y0+:3>+<,)_E'-!-
MX@(@C^`"&HWB%!"!X@Q`C>(!:HWB`S"2YQ!@AN(`,)/E`#"!Y0`PH..X_/_K
MK""?Y0(0H..H,)_E`B"/X`4`C>@#,(_@!`"@X0$@`>.[_/_K!`"@X0$JH.,&
M$*#AL?S_ZP!`4.(`4*"S"@``N@%PA.('`*#AC?S_ZP!04.(%```*!S"@X080
MH.$$(*#A5_S_ZP`PH.,$,,7G2""?Y0(:C>(T,)_E%!"!X@(@C^`#,)+G`""3
MY0`PD>4"(#/@`#"@XP,``!H%`*#A`MJ-XAS0C>+P@+WH>/S_ZWQ\```<`0``
M]"H``.`J``#4>P``N/__ZO!/+>D!RJ#C#,!-X```C.6H``SE3"*?Y4+=3>)(
M,I_E!-!-X@(@C^``4*#A/`*?Y4'-C>(08(WB/,",X@,PDN<!<*#A``"/X`80
MH.$`,)/E`#",Y0`PH..\_?_K+Q"@XP!`H.$%`*#A`/S_ZP``4.,2```*^`&?
MY040H.$``(_@=_[_ZP4`H.'H(9_E01V-XM0QG^4\$('B`B"/X`,PDN<`()/E
M`#"1Y0(@,^``,*#C:P``&D+=C>($T(WB\(^]Z`<`H.$T_/_KK!&?Y0$0C^!#
M_/_K`*!0XN7__PJ<@9_E@)"-X@`PVN4$<$GB"("/X```4^,J```*``!4XP0`
M``H*$*#A!`"@X6C\_^L``%#C&@``"@H`H.$$_/_K`2!`X@(@BN``,*#A`@!:
MX0"PH.$`,(HP`!"@,P$`BC(#```Z"```Z@,`4.$`$,/E`@``"@$@<^4O`%+C
M^?__"@H`H.'R^__K`+"@X04`H.'O^__K"P"`X`(`@.(!"E#C%0``.OP0G^4`
M`*#C`1"/X!3\_^L`H%#BMO__"@`PVN4``%/CU/__&@``5.,I```*`##4Y2X`
M4^,#```*R*"?Y0&PH.,*H(_@Y___Z@$PU.4``%/CZO__"O?__^JL()_E"#"@
MX0(0H.,'`*#A`B"/X`"@C>4$((WE`2J@XPA0C>43_/_K!A"@X0<`H.&Y^__K
M``!0X]K__QI@,!GE#SH#X@()4^/6__\:`1"@XP<`H.&M^__K``!0X]'__QI0
M`)_E!Q"@X0``C^`&_O_K!P"@X=+[_^N,___J.*"?Y0&PH.,*H(_@P/__ZMC[
M_^MH>P``'`$````J``#4*0``]'H``)0I``!X*0``V"@``(`H``"$*```$"@`
M`.0G```00"WI`<J@XPS`3>#X#XSE+Q"@XP!`H.%]^__K``!0XP%`@!($`*#A
M$$"]Z+3[_^KP0"WI`<J@XPS`3>```(SE(``,Y7PAG^4!VDWB>#&?Y0S03>("
M((_@`1J-X@00@>(`4%#B`S"2YP`PD^4`,('E`#"@XS$```H`,-7E``!3XRX`
M``J#^__K`2"`X@!`H.$!"E+C.@``B@1PC>(%$*#A!P"@X8'[_^L!,$3B`S"'
MX`,`5^$)```J!`"'X`4@C>(`$*#C`@``Z@(`4.$`$,#E`@``"@$P<.4O`%/C
M^?__"B\0H.,'`*#A2?O_ZP``4.,E```*!P!0X0`PH(,`,,"%!P"@X7[[_^O`
M()_E`1J-XK0PG^4$$('B`B"/X`,PDN<`()/E`#"1Y0(@,^``,*#C(P``&@':
MC>(,T(WB\("]Z(P@G^4!&HWB?#"?Y000@>("((_@`S"2YP`@D^4`,)'E`B`S
MX``PH.,5```:9`"?Y0``C^`!VHWB#-"-XO!`O>A?^__J``"@X][__^I(()_E
M`1J-XC`PG^4$$('B`B"/X`,PDN<`()/E`#"1Y0(@,^``,*#C`@``&B``G^4`
M`(_@Z___ZEK[_^N@>```'`$``-QW``"D=P``."8``%AW``#L)0``<$`MZ0'*
MH.,,P$W@Z`^,Y<0DG^4(T$WBP#2?Y0U`H.$"((_@N&2?Y;A4G^4&8(_@`S"2
MYP50C^``,)/E!#"-Y0`PH.,$$*#A!@"@X:#\_^L``%#C#@``"HPTG^4``)WE
M`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[
M__\:Z___ZE1DG^4&8(_@!!"@X08`H.&*_/_K``!0XPX```HT-)_E``"=Y0,P
ME><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___
M&NO__^H`9)_E!F"/X`00H.$&`*#A=/S_ZP``4.,.```*W#.?Y0``G>4#,)7G
M`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK
M___JK&.?Y09@C^`$$*#A!@"@X5[\_^L``%#C#@``"H0SG^4``)WE`S"5YP`0
MD^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___
MZEACG^4&8(_@!!"@X08`H.%(_/_K``!0XPX```HL,Y_E``"=Y0,PE><`$)/E
M`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^H$
M8Y_E!F"/X`00H.$&`*#A,OS_ZP``4.,.```*U#*?Y0``G>4#,)7G`!"3Y0`Q
M@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___JL&*?
MY09@C^`$$*#A!@"@X1S\_^L``%#C#@``"GPRG^4``)WE`S"5YP`0D^4`,8'@
M!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZEQBG^4&
M8(_@!!"@X08`H.$&_/_K``!0XPX```HD,I_E``"=Y0,PE><`$)/E`#&!X`0@
MD^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^H(8I_E!F"/
MX`00H.$&`*#A\/O_ZP``4.,.```*S#&?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E
M`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___JM&&?Y09@C^`$
M$*#A!@"@X=K[_^L``%#C#@``"G0QG^4``)WE`S"5YP`0D^4`,8'@!""3Y0`A
M@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZF!AG^4&8(_@!!"@
MX08`H.'$^__K``!0XPX```H<,9_E``"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G
M``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^H,`9_E!!"@X0``C^"O
M^__K`!!0X@(```KX`)_E``"/X&_\_^OP`)_E!!"@X0``C^"F^__K`!!0X@(`
M``K<`)_E``"/X&;\_^O4`)_E!!"@X0``C^"=^__K`!!0XA(```K``)_E``"/
MX%W\_^NX()_E9#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@XQ```!J8$)_E
MF`"?Y0$0C^```(_@"-"-XG!`O>A-_/_JA`"?Y000H.$``(_@A/O_ZP`04.+I
M__\*<`"?Y0``C^!$_/_KY?__ZASZ_^OP=@``'`$``,@E``#<=@``&`$``&@E
M```8)0``S"0``'PD```T)```>",``)`C``!$(P``^"(``"0B``!0(@``)"(`
M`$`B``!`(@``/"(``&`A``"8<@``&"(``'PA``#H(0``;"$```3@+>4!RJ#C
M#,!-X/`/C.5X()_E#-!-XG0PG^4-$*#A`B"/X&P`G^4``(_@`S"2YP`PD^4$
M,(WE`#"@XU/[_^L``%#C!0``"@``T.4``%#C,`!0$P$`H`,``*`3`0`@XC0@
MG^4H,)_E`B"/X`,PDN<`()/E!#"=Y0(@,^``,*#C`0``&@S0C>($\)WDWOG_
MZZQQ```<`0``S"```&1Q```/`"WI`<J@XP3@+>4,P$W@\`^,Y4@0G^4,T$WB
M1""?Y10PC>(!$(_@/`"?Y0`PC>4XP)_E``"/X`(@D><"$*#C`""2Y00@C>4`
M(*#C$""=Y0S`D.<``)SEKOG_Z_\`H./]^?_K"'$``!P!``#X<```(`$``/!'
M+>D`P*#C`>B@XPS`3>`.X$S@`<I,X@``C.4.`%SA^___&C@`#N4!V$WB&-!-
MXLS!G^4"$*#C`>B-X@S`C^`4X([B`!"-Y0`@H..T$9_E`#"@XP"`H.$!$)SG
M`!"1Y0`0CN4`$*#CG/G_ZP$`4.(`<,'BQR^@X0(P(.`",%/@<S#_Y@(P(^`"
M,%/@`B#"X`-@4.`"<,?@``!7XS@``+H(D*#C`9!`XPQ0C>(`$*#C!B"@X0<P
MH.$(`*#A`!"-Y8;Y_^L!`''C`0!P`ST```H)(*#A!1"@X0@`H.%)^?_K`0!P
MXRD```H'`%#C'P``F@@`0.(`0)7@'```*@BAG^4*H(_@`@``Z@%`1.(%`%3A
M%@``.@@@H.,*$*#A!`"@X1'Y_^L``%#C]O__&@5`1.`&`)3@Q!^GX-`@G^4!
MR(WBP#"?Y13`C.("((_@`S"2YP`@D^4`,)SE`B`SX``PH.,F```:`=B-XAC0
MC>+PA[WH`6A6X@!PQ^(!`'?C`0AV`\G__QH``.#C`!"@X>G__^I\()_E`1B-
MXF@PG^44$('B`B"/X`,PDN<`()/E`#"1Y0(@,^``,*#C$```&E0`G^4``(_@
M</__ZTP@G^4!&(WB,#"?Y100@>("((_@`S"2YP`@D^4`,)'E`B`SX``PH.,"
M```:)`"?Y0``C^!B___K//G_ZWQP```<`0``]'```'!O```8;P``L!X``.!N
M``!H'@``$$`MZ0'*H.,,P$W@^`^,Y>WX_^L0^?_K``!0XP``D!40@+WH\$`M
MZ0'*H.,,P$W@T`^,Y0`AG^4<T$WB_#"?Y1`0C>("((_@]'"?Y0!0H.$'<(_@
M`S"2YP<`H.$`,)/E%#"-Y0`PH.-Z^O_K`$!0X@(```H`,-3E``!3XPX``!K`
M()_EM#"?Y0(@C^`#,)+G`""3Y10PG>4"(#/@`#"@XR0``!J@`)_E!1"@X0``
MC^`<T(WB\$"]Z"O[_^H%`*#AW/C_ZP!@H.$$`*#AV?C_ZP!@AN`"8(;B!@"@
MX?GX_^MH,)_E:!"?Y08@H.$#,(_@"$"-Y0$0C^``4(WE!!"-Y0(0H.,`0*#A
M$OG_ZT0@G^4H,)_E`B"/X`,PDN<`()/E%#"=Y0(@,^``,*#C!!"@`0<`H`'=
M__\*Y?C_ZU!N```<`0``"!X```QN``"L'0``@!P``'0<``"`;0``\$\MZ0'*
MH.,,P$W@X`Z,Y?S03>+4)I_EU!:?Y9!PC>("((_@S,:?Y2`@C>4!$(_@Q":?
MY0S`C^`<`(WE##",XAA@C.*H0(WB!."@X3#`C.("()'GG%"-XBB`C>(`()+E
M]""-Y0`@H.,'`)/H!P"'Z`\`MN@/`*[H`P"6Z`,`CN@'`)SH!P"%Z'`&G^4(
M$*#A``"/X!CZ_^L`8%#B`@``"@`PUN4``%/C*0$`&F_X_^N2^/_K``!0XX@`
M``H`8)#E``!6XX4```H&`*#A@OC_ZX`@H.$!L(+B"P"@X:+X_^L`,-;E`%"@
MX0``4^,D`(T5"Y"@$20!``H$II_E"J"/X`L@H.$%`*#A`#"-Y0(0H.,*,*#A
M`E"%XK?X_^L+`%GA"S"@,0DPH"$+($/@`K!+X@.P2^`"((+B`P!2X0*PBR`#
ML(LP`3#VY0``4^/K__\:L`6?Y0``C^`"``#J!`"TY0``4.,-```*"!"@X>+Y
M_^L`4%#B^/__"@`PU>4``%/C]?__"A/Z_^L``%#C\O__"@4`H.%K^/_K`$!0
MX@T``!H$0)?D``!4XPH```H`,-3E``!3X],```H$`*#A!?K_ZP``4./U__\*
M!`"@X5WX_^L`0%#B\?__"@0`H.$^^/_K)&"=Y0!0H.$&`*#A.OC_ZP!0A>`!
M6X7B!%"%X@4`H.%9^/_K`#6?Y0!PH.$%(*#A`S"/X`@PC>7P-)_E`A"@XPQ@
MC>4#,(_@!#"-Y>`TG^4`0(WE`S"/X&_X_^L'':#C!P"@X=SW_^L!`'#C`P``
M&H?X_^L`$)#E$0!1XZ\``!H($*#A!P"@X4[X_^L`0%#B,0``"IP4G^4@`)WE
M'#"=Y90DG^4!`)#G`A"@XP`PD^4"((_@`'"-Y0``D.4X^/_K`&"@XW0DG^5,
M-)_E`B"/X`,PDN<`()/E]#"=Y0(@,^``,*#C_```&@8`H.'\T(WB\(^]Z!OX
M_^L`8%#BBP``&@0`E>0``%#C"0``"@@0H.&!^?_K`#!0XOC__PH`,-/E``!3
MX_+__QH$`)7D``!0X_7__QH(9)_E#0"@XPV0H.,1^/_K!F"/X`!0H.$)L*#A
M4S"@XR0`C>5N___J.#"=Y0\Z`^(!"5/CR?__&D!@G>7&]__K``!6X<7__QHX
M,)WE4S#HYP<-4^/!__\:!0"@X?WW_^L`8*#AJ`.?Y0@0H.$``(_@7/G_ZP`0
M4.*````*'#"=Y0``D^6#^__K`*!0XGL```H*`*#AR_?_ZP"0H.$#`%#C#```
MFFP3G^4$`$#B``"*X`$0C^`C^/_K``!0XP4``!H$`%GCIP``"@F0BN`%,%GE
M+P!3XZ,```H\`Y_E"!"@X0``C^`_^?_K``!0XP,```H`,-#E,`!3XP``4Q,$
M```:"@"@X0`0H..]]__K`(!0XET``!H$$Y_E!`.?Y0$0C^``HY_E``"/X/R2
MG^7R^?_K^(*?Y=_W_^OT,I_E]"*?Y0(0H.,#,(_@!#"-Y>@RG^4"((_@"`"-
MY08`H.$,((WE`S"/X`4@H.$`<(WE"J"/X`F0C^`(@(_@VO?_ZPX``.KV]__K
M`#"0Y1$`4^,/```:R/?_ZP%`A.(`P*#A"C"@X04@H.$"$*#C!@"@X1"0C>4,
M0(WE@!&-Z,KW_^L'':#C!@"@X3?W_^L!`'#CZ___"@<`H.%C]__K8`*?Y080
MH.$``(_@Q/G_ZP8`H.%P_O_K9O__Z@-`H.$Q___J<??_ZX`@H.$!L(+B[?[_
MZO`AG^4@`)WE'#"=Y0(`D.<`,)/E'"*?Y000C>4"$*#C`'"-Y0(@C^```)#E
MC/?_ZU+__^I;_O_K_"&?Y9@QG^4"((_@`S"2YP`@D^7T,)WE`B`SX``PH.-/
M```:!@"@X?S0C>+P3[WH;O?_ZB0`C>7N_O_J'#"=Y0"@D^6`___JMOW_ZP"0
MH.$`H%'BG?__N@H@4.(`L*#C`##*X@@`H.$`L(WE9??_Z\`0C>(&(*#C"`"@
MX1P0C>4J]__K'!"=Y08`4.,U```:P""=Y4,\H.-!,T3C`P!2X8K__QJT(-'A
M2#4$XP,`4N&&__\:,B!9X@@`H.$`,,KBR$"-X@"PC>5.]__K"`"@X2@@H.,$
M$*#A%/?_ZR@`4.,@```:'#&?Y04@H.$8$9_E!@"@X0,PC^`$,(WE##&?Y0$0
MC^`(0(WE#!"-Y0,PC^`"$*#C`'"-Y?"PS>5@]__KF?__ZAPPG>4$D)/E``!9
MXU?__PH)`*#A$??_ZP,`4.-3__^:R!"?Y00`0.(``(G@`1"/X&KW_^L``%#C
M":"@`4O__^HM]__KJ""?Y3`PG^4"((_@`S"2YP`@D^7T,)WE`B`SX``PH./U
M__\:B`"?Y0``C^!&_?_K&&T```QM``!`;@``'`$``&`;```8'```6!L``!@;
M``#T&0``Z!H``"`!``#<&@``C&H``.@9``!`&@``!!H``&P8``"L&```*!@`
M`(09```H&@``D!<``,`7``!(&@``@!D``/P6``#L%P``V&<``,P5``!4&```
M@!<``"@7``!P9@``\!8``'!`+>D!RJ#C#,!-X.@/C.6@$I_E"-!-XIPBG^5P
M3QWN`1"/X)0RG^64`I_E#6"@X0(@D><``(_@#1"@X0`@DN4$((WE`""@XP,P
MG^<#4)3G9##5Y64PQ>5)^/_K``!0XP(```H`,-#E,0!3XV\```J.^__K3#*?
MY0,PG^<#4)3G1!*?Y04`H.$#,*#C`2"@XP$0C^"(]O_K`%!0X@P```HH,I_E
M#""5Y0,PG^<`$)+E'"*?Y0,`E.<"((_@A/;_ZPPPE>4`$)/E"#"1Y0$%$^-A
M```:_!&?Y0,PH.,!(*#C`1"?YP$`E.?L$9_E`1"/X''V_^L`4%#B"P``"MP!
MG^4&$*#A``"/X!WX_^L,,)7E`"!0X@`0D^5(```*P#&?Y0,PG^<#`)3G:?;_
MZ[0QG^4!(*#CL!&?Y0$0C^`#,)_G`U"4YP4`H.%,]O_KG"&?Y0`0H.$%`*#A
M`B"/X)!1G^5;]O_KC#&?Y8P!G^4%4(_@`""@XP80H.$``(_@`S"?YP,PE.=E
M(,/E_??_ZP!`H.$``%#C"`"%Y1(```H'':#C!`"@X2;V_^L!((#B`#"@X1$`
M4.,!`%(3+0``BCPAG^7T,)_E`B"/X`,PDN<`()/E!#"=Y0(@,^``,*#C(P``
M&@C0C>)P@+WH!`"5Y:7]_^L`0*#A``!0XP@`A>7F__\:_""?Y;`PG^4"((_@
M`S"2YP`@D^4$,)WE`B`SX``PH.,2```:W`"?Y0``C^"%]O_K`3#0Y0``4^.0
M__\*B___ZL0PG^4!`*#A`S"/X`0PD^4`$)/E(OG_Z[/__^JL,)_E`S"?YP,`
ME.=;]O_KF/__ZF+V_^N8$)_E/""?Y0$0C^`"()'G`!"2Y00@G>4!$#+@`""@
MX_7__QIX$)_E!""@X70`G^4!$(_@``"/X`00D>4`$)'E8_;_ZZ!E```<`0``
MA&8``)@4``!49@``$!8``"1F``#L%0``[&4``,`5``"`$P``K&4``)!E``!X
M%0``<!4``-1E``!090```!,``/QC``"X8P``J!0``/AD``!L9```1&,``)QD
M``!,%```,$`MZ0'*H.,,P$W@Z`^,Y0PAG^4`0*#A"#&?Y0S03>("((_@``&?
MY0T0H.$``(_@`S"2YP`PD^4$,(WE`#"@XX?W_^L`,%#B"@``"@`PT^4P`%/C
M``!3$P$PH`,`,*`3`3`CX@``5.,`,*`#`3`#$@``4^,*```:L""?Y:0PG^4"
M((_@`S"2YP`@D^4$,)WE`B`SX``PH.,@```:#-"-XC"`O>@`,-3E``!3X_'_
M_PH$`*#A2/K_ZR\0H.,`4*#AOO7_ZP``4.,!4(`2!0"@X?;U_^M8$)_E!""@
MXP$0C^`B]O_K``!0X^+__QI$()_E,#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@
M`#"@XP,``!H$`*#A#-"-XC!`O>BS]__J[O7_ZX!B```<`0``G!$``"!B``"T
M$0``K&$``/!/+>D!RJ#C#,!-X(`.C.403)_E5]]-X@SLG^40,(WB!$"/X`,0
MH.$<((#B7,"`X@[@E.<`X)[E5.&-Y0#@H..-"F+TX`#P\P(`7.'="D'T^O__
M&@9+W>T(.]WM$1Z-X@I;W>T```#JLE%B\@@KT^TT((/B`!O3[0@P@^(!`%/A
MCP=B]+$10O.Q`4#SM`%`\[-!8_*U,67RH!A@\C``X?.P`6'R#@O#[>[__QH@
M0)#IF3D'XX(Z1>,,0)#E$,"=Y0!@D.4$$"[@$'"0Y0/`C.`%$`'@99&@X0?`
MC.`$$"'@%""=Y>;-C.`!P(S@"1`NX`,@@N`&$`'@9H&@X00@@N`8H)WE#A`A
MX`(0@>`(("G@`Z"*X`P@`N`.H(K@"2`BX`H@@N`<H)WE[!V!X&S!H.$#H(K@
M":"*X`R0*.`!D`G@X2V"X`B0*>!A$:#A"I")X""@G>7BG8G@`Z"*X`B@BN`!
M@"S@`H`(X&(AH.$,@"C@`K`AX`J`B.`DH)WE";`+X.F-B.`#H(K@#*"*X&G!
MH.$!D"O@#+`BX`J0B>`HH)WE"+`+X.B=B>`#H(K@`:"*X&@1H.$"@"O@`;`L
MX`J`B.`LH)WE";`+X.F-B.`#H(K@`J"*X&DAH.$,D"O@`K`AX`J0B>`PH)WE
M"+`+X.B=B>`#H(K@#*"*X&C!H.$!@"O@#+`BX`J`B.`TH)WE";`+X.F-B.`#
MH(K@`:"*X&D1H.$"D"O@`;`LX`J0B>`XH)WE"+`+X.B=B>`#H(K@`J"*X&@A
MH.$,@"O@`K`AX`J`B.`\H)WE";`+X.F-B.`#H(K@#*"*X&G!H.$!D"O@#+`B
MX`J0B>!`H)WE"+`+X.B=B>`#H(K@`:"*X&@1H.$"@"O@`;`LX`J`B.!$H)WE
M";`+X.F-B.`#H(K@`J"*X&DAH.$,D"O@`K`AX`J0B>!(H)WE"+`+X.B=B>`#
MH(K@#*"*X&C!H.$!@"O@#+`BX`J`B.!,H)WE";`+X.F-B.`#H(K@`:"*X&D1
MH.$"D"O@`;`LX`J0B>!0H)WE"+`+X.B=B>`#H(K@`J"*X&@AH.$,@"O@"H"(
MX%2@G>7IC8C@`Z"*X`S`BN`"H"'@":`*X&F1H.$!H"K@#*"*X%C`G>7HK8K@
M`\",X`'`C.`)$"+@"!`!X&B!H.$"$"'@#!"!X%S`G>7J'8'@`S",X`C`*>`*
MP`S@`C"#X`G`+.!@()WE`\",X*$[#N/9/D;C:J&@X0,@@N#AS8S@"9""X&0@
MG>4#((+@""""X`J`*.`!@"C@"(")X&&1H.%H$)WE[(V(X`,0@>`*$('@":`J
MX`R@*N!LP:#A"B""X.BM@N!L()WE`R""X`D@@N`,D"G@")`IX&B!H.$)$('@
MZIV!X'`0G>4#$('@#!"!X`C`+.`*P"S@:J&@X0P@@N#IS8+@=""=Y0,@@N`(
M((+@"H`HX`F`*.!ID:#A"!"!X.R-@>!X$)WE`Q"!X`H0@>`)H"K@#*`JX&S!
MH.$*((+@Z*V"X'P@G>4#((+@"2""X`R0*>`(D"G@:(&@X0D0@>#JG8'@@!"=
MY0,0@>`,$('@",`LX`K`+.!JH:#A#"""X.G-@N"$()WE`R""X`@@@N`*@"C@
M"8`HX&F1H.$($('@[(V!X(@0G>4#$('@"A"!X`F@*N`,H"K@;,&@X0H@@N#H
MK8+@C""=Y0,@@N!JL:#A"2""X`R0*>`(D"G@:(&@X0D0@>#JG8'@D!"=Y0,0
M@>`,$('@",`LX`K`+.`,((+@Z:V"X)0@G>4#((+@""""X`N`*.`)P"C@:9&@
MX0P0@>"8P)WEZHV!X`D0*^`#P(S@"A`AX`O`C."<L)WE:J&@X0$@@N`#L(O@
MZ!V"X`FPB^`*D"G@""`IX&B!H.$"P(S@H""=Y>&=C.`#((+@"B""X`B@*N`!
MP"K@I*"=Y6$1H.$,P(O@`Z"*X.G-C.`(H(K@`8`HX`F`*.!ID:#A""""X+"P
MG>7LC8+@J""=Y0,@@N`!((+@"1`AX`P0(>!LP:#A`:"*X`P0*>#HK8K@"!`A
MX`$@@N!H@:#AZAV"X*P@G>4#((+@"#`LX`HP(^`)((+@`R""X-P\"^,;/TCC
M:I&@X0.PB^`)H(CA#+"+X+3`G>7A+8+@`:`*X`/`C.!A$:#A",",X`F`".`(
MH(KA`8")X0N@BN`"@`C@`;`)X.*MBN`+@(CAO+"=Y0R`B."XP)WE8B&@X0.P
MB^`#P(S@`;"+X`F0C.`"P('A"L`,X`(0`>`!P(SAZHV(X`G`C.#`D)WE:J&@
MX>C-C.`*$(+A`Y")X`*0B>`($`'@"B`"X&B!H.$"$('A""`*X`NP@>`($(KA
M#!`!X.R]B^`"$('A;,&@X0D0@>#$D)WE#""(X>L=@>`#D(G@"R`"X`J0B>#(
MH)WE:[&@X0.@BN`(H(K@#(`(X`@@@N%A@:#A"9""X`L@C.'AG8G@`2`"X,P0
MG>4#$('@#!"!X`O`#.`,P(+A"""+X0K`C.`)(`+@"*`+X.G-C.`*((+AT*"=
MY6F1H.$!$(+@`Z"*X`D@B.$+H(K@U+"=Y0P@`N#L'8'@`["+X&S!H.$(L(O@
M"8`(X`@@@N%A@:#A"J""X`P@B>'AK8K@`2`"X-@0G>4#$('@"1"!X`R0">`)
MD(+A""",X0N0B>`*(`+@"+`,X.J=B>`+((+AX+"=Y0$0@N#<()WE:J&@X0.P
MB^`#((+@"+"+X`S`@N`*((CA"2`"X`J`".#I'8'@"(""X6F1H.$,@(C@"2"*
MX6'!H.'AC8C@`2`"X.00G>4#$('@"A"!X`F@"N`*H(+A#"")X0N@BN`((`+@
M#+`)X.BMBN`+((+A[+"=Y0$0@N#H()WE:(&@X0.PB^`#((+@#+"+X`F0@N`(
M((SA"B`"X`C`#.#J'8'@#"""X6JAH.$)D(+@"B"(X6'!H.'AG8G@`2`"X/`0
MG>4#$('@"!"!X`J`".`(@(+A#""*X0N`B.`)(`+@#+`*X.F-B.`+((+A^+"=
MY0$@@N#T$)WE:9&@X0.PB^`#$('@#+"+X`J@@>`)$(SA"!`!X`G`#.`,P('A
MZ"V"X`K`C.#\H)WE:(&@X>+-C.`($(GA`S"*X`DP@^`"$`'@")`)X&(AH.$"
MH(CA"1"!X0L0@>`,H`K@`K`(X.P=@>!LD:#A"Z"*X0#!G>4#H(K@UC$,XV(Z
M3.,#P(S@X:V*X`B`C.`$P9WE8;&@X0/`C.`"P(S@"2`BX`$@(N`($9WE`H"(
MX`L@*>#JC8C@"B`BX`,0@>!JH:#A"1"!X`+`C.`*D"O@Z,V,X`B0*>`,(9WE
M"1"!X&B!H.'LG8'@$!&=Y0,@@N`#$('@"R""X`H0@>`(H"K@#*`JX&S!H.$*
M((+@Z:V"X!0AG>4#((+@""""X`R`*.`)@"C@:9&@X0@0@>#JC8'@&!&=Y0,0
M@>`,$('@"<`LX`K`+.!JH:#A#"""X.C-@N`<(9WE`R""X`D@@N`*D"G@")`I
MX&B!H.$)$('@[)V!X"`1G>4#$('@"A"!X`B@*N`,H"K@;,&@X0H@@N#IK8+@
M)"&=Y0,@@N`(((+@#(`HX`F`*.!ID:#A"!"!X.J-@>`H$9WE`Q"!X`P0@>`)
MP"S@"L`LX&JAH.$,((+@Z,V"X"PAG>4#((+@"2""X`J0*>`(D"G@:(&@X0D0
M@>#LG8'@,!&=Y0,0@>`*$('@"*`JX`R@*N!LP:#A"B""X.FM@N`T(9WE`R""
MX`@@@N`,@"C@"8`HX&F1H.$($('@ZHV!X#@1G>4#$('@#!"!X`G`+.`*P"S@
M:J&@X0P@@N#HS8+@/"&=Y0,@@N`)((+@"I`IX`B0*>!H@:#A"1"!X$"1G>7L
M'8'@`Y")X`J0B>`(H"K@#*`JX&S!H.$*((+@1*&=Y>$M@N`#H(K@"*"*X`R`
M*.`!@"C@81&@X0B0B>!(@9WEXIV)X`.`B.`,@(C@`<`LX`+`+.!B(:#A#,"*
MX`)PA^#IS8S@$'"`Y6SACN`(X(WE3.&=Y0,PCN`&,(/@`3"#X`(0(>`)$"'@
M:9&@X0$0B.`)("+@[!V!X`P@(N`",(/@"4"$X.$]@^`,0(WE`#"-Y0%0A>`\
M()_E!%"-Y=\*;?0"((_@*#"?Y8\*0/0#,)+G`""3Y50QG>4"(#/@`#"@XP$`
M`!I7WXWB\(^]Z-[R_^M$80``'`$``&Q5```00"WI`<J@XPS`3>#X#XSE8`"@
MX]+R_^L'"]_M""O?[0D[W^T`(*#C!`O`[1@@@.5<((#EWRI`]!"`O>CPX=+#
M``````$C16>)J\WO_MRZF'94,A#P1RWI`<J@XPS`3>#@#XSE%#"0Y0)0H.&"
M(:#A`&"@X0(PD^`8()#E%#"`Y0%PH.%<`)#E`2""(J4N@N`8((;E``!0XQR@
MA@(D```:/P!5XQP``-I`@$7B)5.@X0&01>('0*#A!5.'X`0PH.$*(*#A0!"$
MX@#@D^40,(/B#,`3Y1`@@N((`!/E!$`3Y0$`4^$$0`+E$.`"Y0S``N4(``+E
M\___&@-`H.$&`*#AL?S_ZP4`5.'K__\:0'"'X@E32.`_@,CC!W"(X`4@H.$'
M$*#A"@"@X6_R_^M<4(;E\(>]Z$!`8.(<H(;B!0!4X0``BN`%0*"A!""@X6;R
M_^M<,);E`S"$X%PPAN5``%/C\(>]&`8`H.$$4$7@!'"'X)7\_^O)___J<$`M
MZ0'*H.,"BRWM#,!-X.@/C.4!0*#A%!"1Y0!0H.$<8(3B,3^_YA@`E.71$>7G
M,"^_YA@K0^P!(('B?S#@XS@`4N,"`(;@`3#&YT$``-I`(&+B`!"@XQCR_^L$
M`*#A>_S_ZRPPA.)0`,#R#PI&]`\*0_0\,(3B#PI#]$PPA.(/!T/T5#"$X@0`
MH.$/AP/T;_S_ZP,PU.4`,,7E!`"@X;(PU.$!,,7E`#"4Y2,TH.$",,7E`#"4
MY0,PQ>4',-3E!##%Y;8PU.$%,,7E!#"4Y2,TH.$&,,7E!#"4Y0<PQ>4+,-3E
M"##%Y;HPU.$),,7E"#"4Y2,TH.$*,,7E"#"4Y0LPQ>4/,-3E###%Y;XPU.$-
M,,7E##"4Y2,TH.$.,,7E##"4Y0\PQ>43,-3E$##%Y;(QU.$1,,7E$#"4Y2,T
MH.$2,,7E$#"4Y1,PQ>4"B[WL<$"]Z.CQ_^HX(&+B`!"@X];Q_^O&___J"#"?
MY0,PC^``,)/E+O+_ZH13```(0"WI"("]Z`$``@`^````00````P```!!````
M$0```$$````G````00```#8```!!`````P```$$````&````00```!4```!!
M````#P```$$````.````00```"8```!!````$P```$$````I````00````(`
M``!!````&0```$$````*````00````L```!!```````````````E<R\E<P``
M`"]P<F]C+W-E;&8O97AE``!$>6YA3&]A9&5R.CIB;V]T7T1Y;F%,;V%D97(`
M26YT97)N86QS.CI005(Z.D)/3U0`````+W!R;V,O)6DO)7,`97AE`"X```!0
M05)?5$5-4`````!005)?4%)/1TY!344`````.@```"5S)7,E<P``+P```%!%
M4DPU3$E"`````%!%4DQ,24(`4$523#5/4%0`````4$523$E/``!005)?24Y)
M5$E!3$E:140`4$%27U-005=.140`4$%27T-,14%.````4$%27T1%0E5'````
M4$%27T-!0TA%````4$%27T=,3T)!3%]$14)51P````!005)?1TQ/0D%,7U1-
M4$1)4@```%!!4E]435!$25(``%!!4E]'3$]"04Q?5$5-4`!005)?1TQ/0D%,
M7T-,14%.`````#$```!L<V5E:R!F86EL960`````<F5A9"!F86EL960`3$1?
M3$E"4D%265]0051(`%-94U1%30``)3`R>``````E<R5S)7,E<P````!P87(M
M`````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E('-U8F1I<F5C=&]R>2`E<R!F
M86EL960@*&5R<FYO/25I*0H``"5S.B!P<FEV871E('-U8F1I<F5C=&]R>2`E
M<R!I<R!U;G-A9F4@*'!L96%S92!R96UO=F4@:70@86YD(')E=')Y('EO=7(@
M;W!E<F%T:6]N*0H``%!!5$@`````<&%R;``````N<&%R`````'-H;W)T(')E
M860``"5S)7-C86-H92TE<R5S```E<R5S=&5M<"TE=25S````)7,E<W1E;7`M
M)74M)74E<P`````8````<&5R;``````P````4$%2.CI086-K97(Z.E9%4E-)
M3TX`````,2XP-C0```!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C=&]R
M>0`````E<SH@8W)E871I;VX@;V8@<')I=F%T92!C86-H92!S=6)D:7)E8W1O
M<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*`````"UE```M+0``55-%4@````!5
M4T523D%-10````!414U01$E2`%1-4``O=&UP`````&5V86P@)V5X96,@+W5S
M<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII9B`P.R`C(&YO="!R=6YN
M:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?7W!A<E]P;#L*;7D@*"10
M05)?34%'24,L("1&24Q%7V]F9G-E=%]S:7IE+"`D8V%C:&5?;F%M95]S:7IE
M*3L*0D5'24X@>PHD4$%27TU!1TE#(#T@(EQN4$%2+G!M7&XB.PHD1DE,15]O
M9F9S971?<VEZ92`](#0[("`@(R!P86-K*").(BD*)&-A8VAE7VYA;65?<VEZ
M92`](#0P.PHD4$M:25!?34%'24,@/2`B4$M<,#`S7#`P-"(["GT*<W5B(&9I
M;F1?<&%R7VUA9VEC"GL*;7D@*"1F:"D@/2!`7SL*;7D@)&-H=6YK7W-I>F4@
M/2`V-"`J(#$P,C0["FUY("1B=68["FUY("1S:7IE(#T@+7,@)&9H.PIM>2`D
M<&]S(#T@*"1S:7IE+3$I("T@*"1S:7IE+3$I("4@)&-H=6YK7W-I>F4[("`@
M("`@(R!.3U1%.B`D<&]S(&ES(&$@;75L=&EP;&4@;V8@)&-H=6YK7W-I>F4*
M=VAI;&4@*"1P;W,@/CT@,"D@>PIS965K("1F:"P@)'!O<RP@,#L*<F5A9"`D
M9F@L("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H*"1005)?34%'24,I.PII
M9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L("1005)?34%'24,I*2`^/2`P*2![
M"G)E='5R;B`D<&]S("L@)&D["GT*)'!O<R`M/2`D8VAU;FM?<VEZ93L*?0IR
M971U<FX@+3$["GT*;7D@*"1P87)?=&5M<"P@)'!R;V=N86UE+"!`=&UP9FEL
M92P@)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%3E9[4$%27T-,14%.?2D@
M>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ0F%S96YA;64[
M"G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I<B`]($9I;&4Z.D)A<V5N
M86UE.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H<7%;4F5M;W9I;F<@9FEL
M97,@:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN9#HZ9FEN9&1E<'1H*'-U
M8B![("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL("1P87)?=&5M<"D["G)M
M9&ER("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M<"`F)B`D7D\@;F4@)TU3
M5VEN,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ5&5M<"@*5$5-4$Q!5$4@
M/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A<V5N86UE.CID:7)N86UE
M*"1T;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I53DQ)3DL@/3X@,"P**3L*
M;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64["G!R:6YT("1T;7`@/#PB
M+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@9&\*<FT@+7)F("<D
M<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@73L@=&AE;@IB<F5A
M:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N90IR;2`G)&9I;&5N
M86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P,"P@)&9I;&5N86UE.PIM
M>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL(#(^)C$@)B(["G-Y<W1E
M;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG<F]U;F0@<')O8V5S<R!T
M;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI.PI]"GT@?0I"14=)3B![
M"DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F:6YE9"`F26YT97)N86QS
M.CI005(Z.D)/3U0["F5V86P@>PI?<&%R7VEN:71?96YV*"D["FUY("1Q=6EE
M="`]("$D14Y6>U!!4E]$14)51WT["FUY("5S>7,@/2`H"G!A=&A?<V5P("`@
M(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G.B<I+`I?97AE("`@("`@
M("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I;BDO(#\@)RYE>&4G
M(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^("]>35-7:6Y\3U,R+R`_
M("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE*"D["F]U=',H<7%;7"1P
M<F]G;F%M92`]("(D<')O9VYA;64B72D["E]S971?<&%R7W1E;7`H*3L*;W5T
M<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM005)?5$5-4'TB72D["FUY
M("@D<W1A<G1?<&]S+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*3L*34%'24,Z
M('L*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@>WT["G5N;&5S<R`H;W!E
M;B!?1D@L("<\.G)A=R<L("1P<F]G;F%M92D@>PIO=71S*'%Q6T-A;B=T(')E
M860@9G)O;2!F:6QE("(D<')O9VYA;64B72D[("`C(&1O;B=T('5S92`D(2!H
M97)E(&%S(&ET(')E<75I<F5S($5R<FYO+G!M"FQA<W0@34%'24,["GT*;7D@
M)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"I?1D@I.PII9B`H)&UA9VEC
M7W!O<R`\(#`I('L*;W5T<RAQ<5M#86XG="!F:6YD(&UA9VEC('-T<FEN9R`B
M)%!!4E]-04=)0R(@:6X@9FEL92`B)'!R;V=N86UE(ETI.PIL87-T($U!1TE#
M.PI]"F]U=',H(D9O=6YD(%!!4B!M86=I8R!A="!P;W-I=&EO;B`D;6%G:6-?
M<&]S(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I
M>F4@+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*<F5A9"!?1D@L("1B=68L(&QE
M;F=T:"@B7#!#04-(12(I.PII9B`H)&)U9B!N92`B7#!#04-(12(@)B8@)&)U
M9B!N92`B7#!#3$5!3B(I('L*;W5T<R@B3F\@8V%C:&4@;6%R:V5R(&9O=6YD
M(BD["FQA<W0@34%'24,["GT*;W5T<RAQ<5M#86-H92!M87)K97(@(B1B=68B
M(&9O=6YD72D["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T
M7W-I>F4L(#`["G)E860@7T9(+"`D8G5F+"`D1DE,15]O9F9S971?<VEZ93L*
M;7D@)&]F9G-E="`]('5N<&%C:R@B3B(L("1B=68I.PIO=71S*")/9F9S970@
M9G)O;2!S=&%R="!O9B!&24Q%<R!I<R`D;V9F<V5T(BD["G-E96L@7T9(+"`D
M;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2`D;V9F<V5T+"`P.PHD
M<W1A<G1?;V9?1DE,15]S96-T:6]N(#T@=&5L;"!?1D@["FUY("5R97%U:7)E
M7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@("`@("`@("`@("`@("`@
M("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B"G=H:6QE("@D8G5F(&5Q
M(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L
M('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA;64@/2`D8G5F.PIO=71S
M*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B+BXN72D["FUY("1C<F,@
M/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO?'P@*2`_("0Q(#H@
M=6YD968["FUY("@D8F%S96YA;64L("1E>'0I(#T@*"1B=68@/7X@;7PH/SHN
M*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(
M+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@*&1E9FEN960H)&5X="D@
M86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I*2!["FUY("1F:6QE
M;F%M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B=68L(#`W-34I.PHD4$%2
M.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@/2`D9FEL96YA;64["B10
M05(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE?2`]("1F=6QL;F%M93L*
M?0IE;'-E('L*)$UO9'5L94-A8VAE>R1F=6QL;F%M97T@/2!["F)U9B`]/B`D
M8G5F+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L;&YA;64L"GT["B1R97%U
M:7)E7VQI<W1[)&9U;&QN86UE?2LK.PI]"G)E860@7T9(+"`D8G5F+"`T.PI]
M"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S96QF+"`D;6]D=6QE*2`]($!?
M.PIR971U<FX@:68@<F5F("1M;V1U;&4@;W(@(21M;V1U;&4["FUY("1I;F9O
M(#T@)$UO9'5L94-A8VAE>R1M;V1U;&5].PIR971U<FX@=6YL97-S("1I;F9O
M.PID96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE?3L*)$E.0WLD;6]D=6QE
M?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B.PII9B`H)$5.5GM005)?0TQ%
M04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W*2D@>PIM>2`D9F@@/2!)
M3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE(")#86XG="!C<F5A=&4@=&5M
M<"!F:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I.PHD9F@M/G!R:6YT*"1I;F9O
M+3Y[8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*<F5T=7)N("1F:#L*?0IE;'-E
M('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1I;F9O+3Y[8W)C?2YP;2(L
M("1I;F9O+3Y[8G5F?2D["B1I;F9O+3Y[9FEL97T@/2`D9FEL96YA;64["F]P
M96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA;64@;W(@9&EE('%Q6T-A;B=T
M(')E860@(B1F:6QE;F%M92(Z("0A73L*<F5T=7)N("1F:#L*?0ID:64@(D)O
M;W1S=')A<'!I;F<@9F%I;&5D.B!C86XG="!F:6YD(&UO9'5L92`D;6]D=6QE
M(2(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E(%!!4CHZ
M2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@17AP;W)T97(Z
M.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["G)E<75I
M<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA;64@/2`H<V]R="!K97ES
M("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H)$E.0WLD9FEL96YA;65]
M(&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII9B`H)&9I;&5N86UE(#U^
M("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA;64["GT*96QS92!["F1O
M("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^("]S:71E8W5S=&]M:7IE
M7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI<W1[)&9I;&5N86UE?3L*
M?0II9B`H)&)U9B!N92`D4$M:25!?34%'24,I('L*;W5T<RAQ<5M.;R!Z:7`@
M9F]U;F0@869T97(@1DE,12!S96-T:6]N(&EN(&9I;&4@(B1P<F]G;F%M92)=
M*3L*;&%S="!-04=)0R`["GT*)'-T87)T7W!O<R`]("AT96QL(%]&2"D@+2`T
M.R`@("`@("`@("`@("`@("`C('-T87)T(&]F('II<`I]"FUY($!P87)?87)G
M<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@)&-A8VAE7VYA;64I.PID
M96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S86YI=&EZ92`H4D554T4@
M;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*:68@*"$D<W1A<G1?<&]S(&]R
M("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@)B8@<VAI9G0I*2!["FUY
M("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P87(*:2`@(&EN<W1A;&Q?
M<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N7W!A<@IV("`@=F5R:69Y
M7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=(&5Q("<M+7)E=7-E)RD@
M>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%55-%?2`]('-H:69T($!!
M4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E:&%V:6]U<@IM>2!`861D
M7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=66S!=(#U^("]>+2A;04E-
M3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF("@D,2!E<2`G22<I('L*
M<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@*"0Q(&5Q("=-)RD@>PIE
M=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G02<I('L*=6YS:&EF="!`
M<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G3R<I('L*)&]U="`]("0R
M.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L92`]("=S:71E)SL*?0IE
M;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G86QL)SL*?0IE;'-I9B`H
M)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS:68@*"0Q(&5Q("=,)RD@
M>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE('%Q6T-A;B=T(&]P96X@
M;&]G(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D,2!E<2`G5"<I('L*)&-A
M8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD["FEF("AM>2`D8VUD(#T@
M)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG4$%27U1%35`G?3L*:6YI
M=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[(E!!4CHZ1&ES=#HZ)&-M
M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@D7RD@
M9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`24Y#+"!`861D7W1O7VEN
M8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER
M92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$["GT*;7D@)'!A
M<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@*&1E9FEN960@)'!A<BD@
M>PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@<7%;0V%N)W0@
M9FEN9"!P87(@9FEL92`B)'!A<B(Z("0A73L*8FQE<W,H)&9H+"`G24\Z.D9I
M;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH("1Z:7`M/G)E861&
M<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R8VAI=F4Z.EII<#HZ05I?
M3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E("(D
M<&%R(ET["GT*;7D@)6UE=&%?<&%R(#T@9&\@>PII9B`H)'II<"!A;F0@;7D@
M)&UE=&$@/2`D>FEP+3YC;VYT96YT<R@G345402YY;6PG*2D@>PHD;65T82`]
M?B!S+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>7%,N*B\O;7,["B1M971A
M(#U^("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I].PII9B`H9&5F:6YE9"`D
M<&%R*2!["F]P96X@;7D@)'!H+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#
M86XG="!R96%D('!A<B!F:6QE("(D<&%R(CH@)"%=.PIM>2`D8G5F.PIR96%D
M("1P:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I<R!N;W0@82!P87(@9FEL
M95T@=6YL97-S("1B=68@97$@)%!+6DE07TU!1TE#.PIC;&]S92`D<&@["GT*
M0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]($E/.CI&:6QE+3YN97<H"B1O
M=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ1FEL93HZ3U]21%=2*"D@
M?"!)3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI(&]R(&1I92!Q<5M#86XG
M="!C<F5A=&4@9FEL92`B)&]U="(Z("0A73L*)&9H+3YB:6YM;V1E*"D["G-E
M96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H9&5F:6YE9"`D<W1A<G1?
M;V9?1DE,15]S96-T:6]N*2!["G)E860@7T9(+"`D;&]A9&5R+"`D<W1A<G1?
M;V9?1DE,15]S96-T:6]N.PI](&5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PHD
M;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!4E]615)"051)37T@86YD
M("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2!["G)E<75I<F4@4$%2.CI&
M:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY
M*%PD;&]A9&5R+"`D,"D["GT*)&9H+3YP<FEN="@D;&]A9&5R*0IO<B!D:64@
M<7%;17)R;W(@=W)I=&EN9R!L;V%D97(@=&\@(B1O=70B.B`D(5T["FEF("@D
M8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(96%V>3L*4$%2.CI(96%V>3HZ7VEN
M:71?9'EN86QO861E<B@I.PII;FET7VEN8R@I.PIR97%U:7)E7VUO9'5L97,H
M*3L*;7D@0&EN8R`](&=R97`@>R`A+T)31%!!3B\@?0IG<F5P('L**"1B=6YD
M;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>V%R8VAL
M:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF:6=[<')I=FQI8F5X<'TI
M.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R96%C:"!`:6YC.PIM>2`E
M9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,;V%D97(Z.F1L7W-H87)E
M9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI8E]E>'0@/2`D0V]N9FEG
M.CI#;VYF:6=[;&EB7V5X='T[("`@("`@("`@(R!86%@@;&EB7V5X="!V<R!D
M;&5X="`_"FUY("5W<FET=&5N.PIF;W)E86-H(&UY("1K97D@*'-O<G0@:V5Y
M<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D;F%M92D["FEF("AD969I;F5D*&UY
M("1F8R`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&ME>7TI*2!["B@D9FEL
M92P@)&YA;64I(#T@*"1K97DL("1F8RD["GT*96QS92!["F9O<F5A8V@@;7D@
M)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<421D:7)<12\H+BHI)'QI
M*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("0Q*3L*;&%S=#L*?0II9B`H
M)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD?"D@>PII9B`H;7D@)')E
M9B`]("1-;V1U;&5#86-H97LD,7TI('L**"1F:6QE+"`D;F%M92D@/2`H)')E
M9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D,2(I('L**"1F:6QE+"`D
M;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0["GT*?0I]"GT*;F5X="!U
M;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D=W)I='1E;GLD;F%M97TK
M*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I;&4@/7X@+UPN7%$D;&EB
M7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C:VEN9R!&24Q%("(E<R(N
M+BY=+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65](#H@)&9I;&4I*3L*;7D@
M)&-O;G1E;G0["FEF("AR968H)&9I;&4I*2!["B1C;VYT96YT(#T@)&9I;&4M
M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PIO<&5N(&UY("1T
M:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE
M(CH@)"%=.PHD8V]N=&5N="`](#PD=&@^.PIC;&]S92`D=&@["E!!4CHZ1FEL
M=&5R.CI0;V13=')I<"T^87!P;'DH7"1C;VYT96YT+"`B/&5M8F5D9&5D/B\D
M;F%M92(I"FEF("$D14Y6>U!!4E]615)"051)37T@86YD("1N86UE(#U^("]<
M+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N="T^
M;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D;F%M92D["GT*)&9H+3YP
M<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H)&YA;64I("L@.2DL"G-P
M<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z.F-O;7!U=&5#4D,S,B@D
M8V]N=&5N="DL("1N86UE*2P*<&%C:R@G3B<L(&QE;F=T:"@D8V]N=&5N="DI
M+`HD8V]N=&5N="D*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@96UB961D960@
M1DE,12!T;R`B)&]U="(Z("0A73L*;W5T<RAQ<5M7<FET=&5N(&%S("(D;F%M
M92)=*3L*?0I]"FEF("@D>FEP*2!["B1Z:7`M/G=R:71E5&]&:6QE2&%N9&QE
M*"1F:"D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2PIO<B!D:64@<7%;17)R;W(@
M=W)I=&EN9R!Z:7`@<&%R="!O9B`B)&]U=")=.PI]"FEF("@D;65T85]P87)[
M8VQE86Y]*2!["B1F:"T^<')I;G0H(EPP0TQ%04XB*3L*?0IE;'-E('L*:68@
M*"%D969I;F5D("1C86-H95]N86UE*2!["FUY("1C='@@/2!$:6=E<W0Z.E-(
M02T^;F5W*#$I.PIS965K("1F:"P@,"P@,#L*)&-T>"T^861D9FEL92@D9F@I
M.PIS965K("1F:"P@,"P@,CL*)&-A8VAE7VYA;64@/2`D8W1X+3YH97AD:6=E
M<W0["GT*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A8VAE7VYA;64L(#`L("1C
M86-H95]N86UE7W-I>F4["B1C86-H95]N86UE("X](")<,"(@>"`H)&-A8VAE
M7VYA;65?<VEZ92`M(&QE;F=T:"`D8V%C:&5?;F%M92D["B1F:"T^<')I;G0H
M)&-A8VAE7VYA;64L"B)<,$-!0TA%(BD["GT*;7D@)&]F9G-E="`]("1F:"T^
M=&5L;"`M(&QE;F=T:"@D;&]A9&5R*3L*)&9H+3YP<FEN="AP86-K*"=.)RP@
M)&]F9G-E="DL"B1005)?34%'24,I.PHD9F@M/F-L;W-E(&]R(&1I92!Q<5M%
M<G)O<B!W<FET:6YG('1R86EL97(@;V8@(B1O=70B.B`D(5T["F-H;6]D(#`W
M-34L("1O=70["F5X:70["GT*>PIL87-T('5N;&5S<R!D969I;F5D("1S=&%R
M=%]P;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q=6ER92!005(["E!!4CHZ2&5A
M=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR97%U:7)E($9I;&4Z.D9I;F0[
M"G)E<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY("1F:"`]($E/.CI&:6QE+3YN
M97<[("`@("`@("`@("`@("`@("`@("`@("`@("`@("`C($%R8VAI=F4Z.EII
M<"!O<&5R871E<R!O;B!A;B!)3SHZ2&%N9&QE"B1F:"T^9F1O<&5N*&9I;&5N
M;RA?1D@I+"`G<B<I(&]R(&1I92!Q<5MF9&]P96XH*2!F86EL960Z("0A73L*
M07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H+7,@7T9(*3L*;7D@)'II<"`]
M($%R8VAI=F4Z.EII<"T^;F5W.PHH)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE
M*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"DI"F]R
M(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E("(D<')O9VYA;64B
M73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H-C0@*B`Q,#(T*3L*<'5S
M:"!`4$%2.CI,:6)#86-H92P@)'II<#L*)%!!4CHZ3&EB0V%C:&5[)'!R;V=N
M86UE?2`]("1Z:7`["F]U=',H(D5X=')A8W1I;F<@>FEP+BXN(BD["FEF("AD
M969I;F5D("1%3E9[4$%27U1%35!]*2![(",@<VAO=6QD(&)E('-E="!A="!T
M:&ES('!O:6YT(0IF;W)E86-H(&UY("1M96UB97(@*"`D>FEP+3YM96UB97)S
M("D@>PIN97AT(&EF("1M96UB97(M/FES1&ER96-T;W)Y.PIM>2`D;65M8F5R
M7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M93L*;F5X="!U;FQE<W,@)&UE;6)E
M<E]N86UE(#U^(&U["EX*+S]S:&QI8B\**#\Z)$-O;F9I9SHZ0V]N9FEG>V%R
M8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E>'1R86-T7VYA;64@/2`D
M,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M/F-A=&9I;&4H)$5.5GM0
M05)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H+68@)&1E<W1?;F%M92`F
M)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E9%-I>F4H*2D@>PIO=71S
M*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I;F-E(&ET(&%L<F5A9'D@
M97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L<V4@>PIO=71S*'%Q6T5X
M=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D97-T7VYA;64B+BXN72D[
M"B@D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E<W1?;F%M92D@/3T@
M07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@<7%;17)R;W(@97AT<F%C
M=&EN9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N86UE(ET["F-H;6]D*#`U-34L
M("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"(["GT*?0I]"GT*=6YL97-S
M("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2!["F1I92`\/"`B+B(@=6YL
M97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP87(@72!;("U)9&ER(%T@
M6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O9W)A;2YP;"!="B0P(%L@
M+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*)$5.5GM005)?4%)/1TY!
M345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`05)'5BD["GT*<W5B($-R
M96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E<75I<F4@1FEL93HZ0F%S
M96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D97AT*2`]($9I;&4Z.D)A
M<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN*B<I*3L*<F5Q=6ER92!&
M:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H)'!A=&@I('5N;&5S<R@M
M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E<G)O<@I]"G-U8B!R97%U
M:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q=6ER92!$>6YA3&]A9&5R
M.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R:6-T.PIR97%U:7)E('=A
M<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@0V%R<#L*<F5Q=6ER92!#
M87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q=6ER92!%>'!O<G1E<CHZ
M2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I<F4@1F-N=&P["G)E<75I
M<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3<&5C.PIR97%U:7)E(%A3
M3&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER92!)3SHZ2&%N9&QE.PIR
M97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R97-S.CI:;&EB.PIR97%U
M:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*<F5Q=6ER
M92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!005(Z.D1I<W0[
M"G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PIR97%U:7)E(%!!4CHZ
M1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@871T<FEB=71E<SL*979A
M;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U:7)E(%=I;C,R('T["F5V
M86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE=F%L('L@<F5Q=6ER92!!
M<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![(')E<75I<F4@5&EE.CI(
M87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E<75I<F4@4&5R;$E/.R!R
M97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@>R!R97%U:7)E('5T9C@@
M?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD969I;F5D("1%3E9[4$%2
M7U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\I('L*)'!A<E]T
M96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY("1P871H("@**&UA<"`D
M14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%35!$25(@5$5-4"!4
M35`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI('L*;F5X="!U;FQE<W,@
M9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD("UW("1P871H.PIM>2`D
M=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P=W5I9"D@/2!G971P=W5I
M9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E9FEN960H)E=I;C,R.CI,
M;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN,S(Z.DQO9VEN3F%M93L*
M?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S97)N86UE(#T@)'!W=6ED
M.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U5315).04U%?2!\?"`D14Y6
M>U5315)]('Q\("=365-414TG.PI]"FUY("1S=&UP9&ER(#T@(B1P871H)'-Y
M<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U<V5R;F%M92D["FUK9&ER
M("1S=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER.PII9B`H)$5.5GM005)?
M0TQ%04Y]*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["GT*96QS92!["F]P
M96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@;W(@9&EE('%Q6T-A;B=T
M(')E860@(B1P<F]G;F%M92(Z("0A73L*:68@*"AM>2`D;6%G:6-?<&]S(#T@
M9FEN9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS965K("1F:"P@)&UA9VEC
M7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N9W1H*")<,$-!0TA%(BDL
M(#`["FUY("1B=68["G)E860@)&9H+"`D8G5F+"!L96YG=&@H(EPP0T%#2$4B
M*3L*:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E96L@)&9H+"`D;6%G:6-?
M<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG=&@H(EPP0T%#2$4B*0HM
M("1C86-H95]N86UE7W-I>F4L(#`["G)E860@)&9H+"`D8G5F+"`D8V%C:&5?
M;F%M95]S:7IE.PHD8G5F(#U^(',O7#`O+V<["B1C86-H95]D:7(@/2`B8V%C
M:&4M)&)U9B(["GT*?0IC;&]S92`D9F@["GT*:68@*"$D8V%C:&5?9&ER*2![
M"B1C86-H95]D:7(@/2`B=&5M<"TD)"(["B1%3E9[4$%27T-,14%.?2`](#$[
M"GT*)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?21C86-H95]D:7(B.PIM:V1I
M<B`D<W1M<&1I<BP@,#<U-3L*)$5.5GM005)?5$5-4'T@/2`D<W1M<&1I<CL*
M;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6>U!!4E]414U0?2!A;F0@
M)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U8B!?<V%V95]A<R!["FUY
M("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`7SL*;7D@)&9U;&QN86UE
M(#T@(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H+64@)&9U;&QN86UE*2![
M"FUY("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B.PIM>2`D9F@["BAO<&5N
M("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE*0HF)B`H<')I;G0@)&9H("1C;VYT
M96YT<RD*)B8@*&-L;W-E("1F:"D*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@
M(B1T96UP;F%M92(Z("0A73L*8VAM;V0@)&UO9&4L("1T96UP;F%M92!I9B!D
M969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE+"`D9G5L;&YA;64I(&]R
M('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D9G5L;&YA;64["GT*<W5B
M(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]04D]'3D%-
M17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O*"XK*2\I('L*)'!R;V=N
M86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*:68@*"1%3E9[4$%27U1%
M35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[4$%27U1%35!]*2`^/2`P
M*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA;64L(')I;F1E>"@D<')O
M9VYA;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II9B`H(21%3E9[4$%27U!2
M3T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@/CT@
M,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R;V=N86UE*2!["G)E='5R
M;B!I9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA;64D<WES>U]E>&5](BD@
M>PHD<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T=7)N.PI]"GT*9F]R96%C
M:"!M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A=&A?<V5P?5Q%+RP@)$5.5GM0
M051(?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!4E]414U0?2!A;F0@)&1I
M<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S+UQ1)'-Y<WM?9&5L:6U]
M7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA
M;64D<WES>U]E>&5](BDL(&QA<W0I"FEF("US("(D9&ER)'-Y<WM?9&5L:6U]
M)'!R;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES
M>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@+7,@(B1D:7(D<WES>U]D
M96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O9VYA;64@>PHD,"`]
M("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345].PII9B`H:6YD97@H
M)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#P@,"D@>PHD<')O9VYA;64@/2`B
M+B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W9"`]("AD969I;F5D
M("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@*"AD969I;F5D("97
M:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z(&!P=V1@*3L*8VAO
M;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN/UQ1)'-Y<WM?9&5L
M:6U]7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6>U!!4E]04D]'3D%-17T@
M/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E;G8@>PII9B`H("1%3E9[
M4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E='5R;CL*?2!E;'-E('L*
M)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF;W(@*'%W*"!34$%73D5$
M(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!344@*2`I('L*9&5L971E
M("1%3E9[(E!!4E\D7R)].PI]"F9O<B`H<7<O(%1-4$1)4B!414U0($-,14%.
M($1%0E5'("\I('L*)$5.5GLB4$%27R1?(GT@/2`D14Y6>R)005)?1TQ/0D%,
M7R1?(GT@:68@97AI<W1S("1%3E9[(E!!4E]'3$]"04Q?)%\B?3L*?0II9B`H
M)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[4$%27T-,14%.?3L*?0I]
M"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*:68@*"1L;V=F:"D@>PIP
M<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP<FEN="`B0%]<;B(["GT*
M?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I9SL*<'5S:"!`24Y#+"!G
M<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF:6=[)%]]+"!Q=R@*87)C
M:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I=&5L:6)E>'`*=F5N
M9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP86-K86=E(&UA:6X["G)E
M<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2.CIF:6YD7W!A<CL*4$%2
M+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A<BYP;#H@0V%N)W0@;W!E
M;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C:"!F:6QE(&]R(&1I
M<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE.PID;R`D<')O9VYA;64[
M"D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+7T58251?7"@H7&0K*5PI
M+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/4B`]("1`(&EF("1`.PI]
M"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2(#U^+UY?5$M?15A)5%]<
M*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@)#HZ7U]%4E)/4CL*,3L*
M7U]%3D1?7PH`<&5R;'AS:2YC`````$-!0TA%```^````00````P```!!````
M$0```$$````G````00```#8```!!`````P```$$````&````00```!4```!!
M````#P```$$````.````00```"8```!!````$P```$$````I````00````(`
M``!!````&0```$$````*````00````L```!!``````````````"!L@&!L+"L
M!P````"$`@&!L`^Q``````"%L@&!L+"N?P````!LA_]_L*H)@&"*_W\!````
M6(W_?["K'H#8CO]_`0```!R1_W^X__]_2)+_?["H`X#LDO]_`0```+2>_W^L
M__]_()__?[#__W\\H?]_L+"H@%BA_W\!````B*+_?["O/H#XJ?]_L*H!@!"M
M_W\!``````````0````0`````0```$=.50```````P````(`````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````````"$
M&@``$!H```$```!+`P```0```%8#```!````8`,```$```!T`P``'0```/P$
M```,````*!,```T```"(3```&0```*R=```;````!````!H```"PG0``'```
M``0```#U_O]OM`$```4```#X"0``!@```$@#```*````'P4```L````0````
M%0`````````#````S)X```(```#@`0``%````!$````7````2!$``!$```!P
M$```$@```-@````3````"````!X````(````^___;P$```C^__]O\`\``/__
M_V\#````\/__;Q@/``#Z__]O#@``````````````````````````````````
M``````````````````````"TG0`````````````T$P``-!,``#03```T$P``
M-!,``#03```T$P``-!,``#03```T$P``-!,``#03```T$P``-!,``#03```T
M$P``-!,``#03```T$P``-!,``#03```T$P``-!,``#03```T$P``-!,``#03
M```T$P``-!,``#03```T$P``-!,``#03```T$P``-!,``#03```T$P``-!,`
M`#03```T$P``-!,``#03```T$P``-!,``#03```T$P``-!,``#03```T$P``
M-!,``#03```T$P``-!,``#03```T$P``-!,``#03```T$P``-!,``#03````
M````````````````````````````````````````````````````````````
M````````&!8````````$H```"E!!4BYP;0H`````;%```(!-````````0$X`
M`$1.``!@4```B$T``&A0````````3%```%10````````=%```$=#0SH@*$=.
M52D@,30N,BXQ(#(P,C4P,C`W`$=#0SH@*&-R;W-S=&]O;"U.1R`Q+C(W+C!?
M<F,Q+C=?8S@W-#-E82D@,30N,BXQ(#(P,C4P,C`W`$$T````865A8FD``2H`
M```%-RU!``8*!T$(`0D""@,,`1($$P$4`14!%P,8`1D!&@(<`2(!`"YS:'-T
M<G1A8@`N;F]T92YG;G4N8G5I;&0M:60`+FEN=&5R<``N9VYU+FAA<V@`+F1Y
M;G-Y;0`N9'EN<W1R`"YG;G4N=F5R<VEO;@`N9VYU+G9E<G-I;VY?<@`N<F5L
M+F1Y;@`N<F5L+G!L=``N:6YI=``N=&5X=``N9FEN:0`N<F]D871A`"Y!4DTN
M97AT86(`+D%232YE>&ED>``N96A?9G)A;64`+FYO=&4N04))+71A9P`N:6YI
M=%]A<G)A>0`N9FEN:5]A<G)A>0`N9'EN86UI8P`N9V]T`"YD871A`"YB<W,`
M+F-O;6UE;G0`+D%232YA='1R:6)U=&5S````````````````````````````
M```````````````````````````````+````!P````(```!T`0``=`$``"0`
M``````````````0`````````'@````$````"````F`$``)@!```9````````
M```````!`````````"8```#V__]O`@```+0!``"T`0``E`$```0`````````
M!`````0````P````"P````(```!(`P``2`,``+`&```%`````P````0````0
M````.`````,````"````^`D``/@)```?!0`````````````!`````````$``
M``#___]O`@```!@/```8#P``U@````0``````````@````(```!-````_O__
M;P(```#P#P``\`\``(`````%`````P````0`````````7`````D````"````
M<!```'`0``#8````!``````````$````"````&4````)````0@```$@1``!(
M$0``X`$```0````6````!`````@```!N`````0````8````H$P``*!,```P`
M``````````````0`````````:0````$````&````-!,``#03``#D`@``````
M```````$````!````'0````!````!@```!@6```8%@``<#8`````````````
M"`````````!Z`````0````8```"(3```B$P```@```````````````0`````
M````@`````$````"````D$P``)!,``#X00`````````````$`````````(@`
M```!`````@```(B.``"(C@``)```````````````!`````````"3`````0``
M<((```"LC@``K(X``'`````,``````````0`````````G@````$````"````
M'(\``!R/```$```````````````$`````````*@````'`````@```""/```@
MCP``(```````````````!`````````"V````#@````,```"LG0``K)T```0`
M``````````````0````$````P@````\````#````L)T``+"=```$````````
M```````$````!````,X````&`````P```+2=``"TG0``&`$```4`````````
M!`````@```#7`````0````,```#,G@``S)X``#0!``````````````0````$
M````W`````$````#`````*````"@``!(```````````````$`````````.(`
M```(`````P```$B@``!(H```%```````````````!`````````#G`````0``
M`#``````````2*```%0```````````````$````!````\`````,``'``````
M`````)R@```U```````````````!``````````$````#``````````````#1
7H`````$``````````````0``````````
