/*
 * Copyright (C) 2021 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "EventDispatcher.h"

#include "ArgumentCoders.h"
#include "Decoder.h"
#include "EventDispatcherMessages.h"
#include "HandleMessage.h"
#if ENABLE(MOMENTUM_EVENT_DISPATCHER)
#include "ScrollingAccelerationCurve.h"
#endif
#include "WebCoreArgumentCoders.h"
#if ENABLE(MAC_GESTURE_EVENTS)
#include "WebGestureEvent.h"
#endif
#if ENABLE(IOS_TOUCH_EVENTS)
#include "WebTouchEvent.h"
#endif
#include "WebWheelEvent.h"
#if HAVE(CVDISPLAYLINK)
#include <WebCore/DisplayUpdate.h>
#endif
#include <WebCore/PageIdentifier.h>
#include <WebCore/RectEdges.h>
#include <optional>

namespace Messages {

namespace EventDispatcher {

#if ENABLE(IOS_TOUCH_EVENTS)

void TouchEvent::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> handled;
    decoder >> handled;
    if (!handled) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*handled));
}

void TouchEvent::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void TouchEvent::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool handled)
{
    encoder.get() << handled;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

} // namespace EventDispatcher

} // namespace Messages

namespace WebKit {

void EventDispatcher::didReceiveMessage(IPC::Connection& connection, IPC::Decoder& decoder)
{
    Ref protectedThis { *this };
    if (decoder.messageName() == Messages::EventDispatcher::WheelEvent::name())
        return IPC::handleMessage<Messages::EventDispatcher::WheelEvent>(connection, decoder, this, &EventDispatcher::wheelEvent);
#if ENABLE(IOS_TOUCH_EVENTS)
    if (decoder.messageName() == Messages::EventDispatcher::TouchEvent::name())
        return IPC::handleMessageAsync<Messages::EventDispatcher::TouchEvent>(connection, decoder, this, &EventDispatcher::touchEvent);
#endif
#if ENABLE(IOS_TOUCH_EVENTS)
    if (decoder.messageName() == Messages::EventDispatcher::TouchEventWithoutCallback::name())
        return IPC::handleMessage<Messages::EventDispatcher::TouchEventWithoutCallback>(connection, decoder, this, &EventDispatcher::touchEventWithoutCallback);
#endif
#if ENABLE(MAC_GESTURE_EVENTS)
    if (decoder.messageName() == Messages::EventDispatcher::GestureEvent::name())
        return IPC::handleMessage<Messages::EventDispatcher::GestureEvent>(connection, decoder, this, &EventDispatcher::gestureEvent);
#endif
#if HAVE(CVDISPLAYLINK)
    if (decoder.messageName() == Messages::EventDispatcher::DisplayWasRefreshed::name())
        return IPC::handleMessage<Messages::EventDispatcher::DisplayWasRefreshed>(connection, decoder, this, &EventDispatcher::displayWasRefreshed);
#endif
#if ENABLE(MOMENTUM_EVENT_DISPATCHER)
    if (decoder.messageName() == Messages::EventDispatcher::SetScrollingAccelerationCurve::name())
        return IPC::handleMessage<Messages::EventDispatcher::SetScrollingAccelerationCurve>(connection, decoder, this, &EventDispatcher::setScrollingAccelerationCurve);
#endif
    if (decoder.messageName() == Messages::EventDispatcher::PageScreenDidChange::name())
        return IPC::handleMessage<Messages::EventDispatcher::PageScreenDidChange>(connection, decoder, this, &EventDispatcher::pageScreenDidChange);
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled message %s to %" PRIu64, IPC::description(decoder.messageName()), decoder.destinationID());
}

} // namespace WebKit
